package edu.uulm.scbayes.experiments.examples

import edu.uulm.scbayes.mln.parsing.ParseHelpers._
import edu.uulm.scbayes.mln.MarkovLogicNetwork
import edu.uulm.scbayes.logic.TruthAssignment
import edu.uulm.scbayes.mln.factorgraph.MLNFactorGraph

/**
 * This object provides several example factor graphs created from MLNs.
 * 
 * Date: 29.03.11
 */

object ExampleGraphs {

  class ExampleMLN(val mlnSource: String, val evidenceSource: String = "") {
    val (signature, formulaList) = parseMLN(mlnSource)
    val evidence = parseEvidence(evidenceSource, signature)
    lazy val mln: MarkovLogicNetwork = MarkovLogicNetwork(formulaList, signature)
    lazy val graph: MLNFactorGraph = MLNFactorGraph.fromMLN(mln)
  }

  def parseString(mlnSource: String): MLNFactorGraph = {
    val (signature, formulaList) = parseMLN(mlnSource)

    MLNFactorGraph.fromFormulas(formulaList, signature)
  }

  def parseWithEvidence(mlnSource: String, evidenceSource: String): MLNFactorGraph = {
    val (signature, formulaList) = parseMLN(mlnSource)
    val evidence = parseEvidence(evidenceSource, signature)

    MLNFactorGraph.fromFormulas(formulaList, signature, evidence)
  }

  def parseOnlyEvidence(mlnSource: String, evidenceSource: String): TruthAssignment = {
    val (signature, _) = parseMLN(mlnSource) //to get the signature
    parseEvidence(evidenceSource, signature)
  }

  private val smokersMLN = """
  friends(Person, Person)
  smokes(Person)
  cancer(Person) //some comment

  Person = {PETER,THOMAS}      /* and a multi
  line comment */

  1 smokes(?x) => cancer(?x)
  2 friends(?x,?y) => (smokes(?x) <=> smokes(?y))
  """
  def smokers: MLNFactorGraph = parseString(smokersMLN)

  private val smokersMLN3P = """
  friends(Person, Person)
  smokes(Person)
  cancer(Person) //some comment

  Person = {PETER,THOMAS,FELIX}

  1 smokes(?x) => cancer(?x)
  2 friends(?x,?y) => (smokes(?x) <=> smokes(?y))
  """
  def smokers3P: MLNFactorGraph = parseString(smokersMLN3P)

  private val smokerEvidence = """
  smokes(PETER)
  friends(PETER, THOMAS)
  friends(THOMAS, PETER)
  """

  def smokersWithEvidence: MLNFactorGraph = parseWithEvidence(smokersMLN,smokerEvidence)
  def smokersEvidence: TruthAssignment = parseOnlyEvidence(smokersMLN, smokerEvidence)

  private val simpleMLN = """
  a(Foo)

  Foo = {A,B,C}

  1 a(?x) => a(C)
  """
  def simple: MLNFactorGraph = parseString(simpleMLN)

  private val hardSmokersMLN = """
  friends(Person, Person)
  smokes(Person)
  cancer(Person)

  Person = {PETER,THOMAS}

  1 smokes(?x) => cancer(?x)
  friends(?x,?y) => (smokes(?x) <=> smokes(?y)).
  """
  def hardSmokers = parseString(hardSmokersMLN)

  val simpleTemporalSource = """
  emotion(Time, Emo!)
  obs_emo(Time, Emo)

  Emo = {BAD, NEUTRAL, GOOD}
  Time = {0 - 9}

  2 emotion(t,NEUTRAL)
  2   emotion(t,e) => emotion(succ(t),e)
  1.5 emotion(t,e) => obs_emo(t,e)
  """
  def simpleTemporal = parseString(simpleTemporalSource)

  val simpleTemporalEvidence = """
  obs_emo(0,NEUTRAL)
  obs_emo(1,NEUTRAL)
  obs_emo(2,GOOD)
  obs_emo(3,GOOD)
  obs_emo(4,GOOD)
  obs_emo(5,NEUTRAL)
  obs_emo(6,NEUTRAL)
  obs_emo(7,NEUTRAL)
  obs_emo(8,BAD)
  obs_emo(9,BAD)
  """

  lazy val simpleTemporalMLN = new ExampleMLN(simpleTemporalSource)
  lazy val simpleTemporalMLNWithEvidence = new ExampleMLN(simpleTemporalSource,simpleTemporalEvidence)

  private val simpleFunctionalMLN = """
  emotion(Time, Emo!)
  obs_emo(Time, Emo)

  Emo = {BAD, NEUTRAL, GOOD}
  Time = {T1,T2}

  2 emotion(t,NEUTRAL)
  1.5 emotion(t,e) => obs_emo(t,e)
  """
  def simpleFunctional = parseString(simpleFunctionalMLN)

  private val pascalCrystalMLN = """
  at(Time,Location!)
  spend(Time)
  spotEntering(Time,Location)

  Location = {ROOM, ELSE}

  Time = {0-10}

  spend(t) => spend(succ(t)).

  !spend(t) ^ !at(t,ROOM) => !spend(succ(t)).

  2 !spend(t) ^ at(t,ROOM) => !spend(succ(t))

  4 !at(t,l) ^ at(succ(t),l) <=> spotEntering(succ(t),l)

  2 at(t,r) => at(succ(t),r)
  """

  def pascalCrystal = parseString(pascalCrystalMLN)

  private val linearSeqMLNsrc = """
  p(A,B!)
  A = {0 - 9}
  B = {T, F}

  p(t,b) <=> p(t + 1,b).
  p(0,T).
  """

  lazy val linearSeqMLN = new ExampleMLN(linearSeqMLNsrc)

  private val socialForce = """
  at(Time, Agent, Location!)

  Location = {0 - 9}
  Time = {0 - 20}
  Agent = {A, B}

  //only move one step at a time
  3 (at(t + 1, a, l) ^ (at(t, a, l + -1) v at(t, a, l) v at(t, a, l + 1)))
  3 (at(t + 1, a, 0) ^ (at(t, a, 0) v at(t, a, 1)))
  3 (at(t + 1, a, 9) ^ (at(t, a, 9) v at(t, a, 8)))

  //two agents don't occupy the same location
  3 !(at(t,a,l) ^ at(t,b,l))
  """
  private val socialForceEvidence = """
  at( 0,A,1)
  at( 2,A,1)
  at( 5,A,2)
  at( 9,A,3)
  at(14,A,4)
  at(20,A,3)

  at( 5,B,3)
  at(10,B,4)
  at(15,B,5)
  at(18,B,4)
  """

  lazy val socialForceMLN = new ExampleMLN(socialForce,socialForceEvidence)

}