package edu.uulm.scbayes.factorgraph

/**
 * Trait that represents hypergraphs.
 * Date: 12.08.11
 */

trait EdgeLabeledHyperGraph[N, E] extends HyperGraph[N] {
  def nodes: Set[N]
  def edges: Set[E]

  def edgeSets = edges map nodesOf

  def nodesOf(edge: E): Set[N]

  def edgesOf(node: N): Set[E] = edges.filter(e => nodesOf(e).contains(node))

  /**
   * Overridden for optimization.
   * @return A set of edges, that include all other edges.
   */
  override def adjacent(n1: N, n2: N) = {
    val factorsOf1 = edgesOf(n1)
    val factorsOf2 = edgesOf(n2)
    if( factorsOf1.size < factorsOf2.size )
      factorsOf1.exists(factorsOf2)
    else
      factorsOf2.exists(factorsOf1)
  }

  def edgeCover: Set[E] = {
    //Note: F1 is subfactor of F2 <=> variablesOf(F1) includes variablesOf(F2)
    // - begin with the largest factor and remove all subfactor
    // - add it to the list and remove all subfactors from the remaining set

    //first tuple elem are the super factors, second elem are the remaining uncovered factors
    val coverIt = Iterator.iterate((Set(): Set[E], edges)){ case (sf,remaining) =>
      val biggestRemaining: Option[E] = if(!remaining.isEmpty) Some(remaining.maxBy(nodesOf(_).size)) else None

      val newRemaining = for(
        bf <- biggestRemaining;
        subFactors = subEdgesOf(bf)
      ) yield remaining -- subFactors

      (sf ++ biggestRemaining.toSeq, newRemaining.getOrElse(Set()))
    }

    val result = coverIt.dropWhile(!_._2.isEmpty).next()._1
//    assert(edges.forall(edge =>
//      result.exists(coverEdge =>
//        nodesOf(edge) subsetOf nodesOf(coverEdge)
//      )), "produced cover is not a cover")
    result
  }

  def subEdgesOf(e: E): Set[E] = edges.filter(sf => isSubEdgeOf(sf,e))

  /** @return true if all nodes of e1 are also nodes of e2. */
  def isSubEdgeOf(e1: E, e2: E) = nodesOf(e1).subsetOf(nodesOf(e2))
}

trait HyperGraph[N] {
  def nodes: Set[N]
  def edgeSets: Set[Set[N]]
  def adjacent(n1: N, n2: N) = edgeSets.exists(e => e(n1) && e(n2))
}