package edu.uulm.scbayes.factorgraph

import edu.uulm.scbayes.probabilities.DiscreteVariable
import edu.uulm.scbayes.util._

/**
 * A factor node for a discrete factor graph. This isn't necessarily discrete, so ...
 *
 *
 * Date: 3/10/11
 */

class TableFactor[V <: DiscreteVariable](assignmentVars: IndexedSeq[V], values: IndexedSeq[Int] => Double)
  extends DiscreteFactor[V] with IndexedFactor {

  val assignmentVariables: IndexedSeq[V] = assignmentVars
  val cpi = new CrossProductIndexer(assignmentVariables.map(_.domainSize))
  val lookup: Array[Double] = (0 until cpi.size).map(i => values(cpi.index2Seq(i))).toArray

  def logFactorByIndex(index: Int): Double = lookup(index)

  def assignmentToIndex(assign: IndexedSeq[Int]): Int = cpi.seq2Index(assign)

  def index2Assignment(index: Int): IndexedSeq[Int] = cpi.index2Seq(index)

  def logFactor(assign: IndexedSeq[Int]): Double = lookup(cpi.seq2Index(assign))

  override def toString: String = {
    assignmentVars match {
      case Seq(x1) => TableFormatter.table1(x1.getRange,(i: Int) => "%.2f".format(logFactor(IndexedSeq(i))))
      case Seq(x1,x2) => TableFormatter.table2(x1.getRange,x2.getRange)(
        (i1, i2) => "%.2f".format(logFactor(IndexedSeq(i1,i2))),
        x1.valueName(_),
        x2.valueName(_)
      )
      case _ => "you don't want to see this"
    }
  }
}

object TableFactor {
  def combineFactors[V <: DiscreteVariable, F <: DiscreteFactor[V]](factors: Set[F]): TableFactor[V] = {
    val variableSuperset: Set[V] = factors.flatMap(_.variables) (collection.breakOut)
    //have it in a fixed order
    val varAssignment = variableSuperset.toIndexedSeq

    /** Take an assignment to varAssignment and turn it into an assignment of theirVars. */
    def reshuffleAssignment(myAss: IndexedSeq[Int], theirVars: IndexedSeq[V]): IndexedSeq[Int] = {
      theirVars.map(varAssignment.zip(myAss).toMap)
    }

    //this function maps assignments to varAssignment-sequence to the sum over
    //the corresponding assignments of factors
    val accValue: IndexedSeq[Int] => Double = { ints =>
      factors
        //map factors to their value under ints
        .map(f => f.logFactor(reshuffleAssignment(ints, f.assignmentVariables)))
        //sum them up
        .sum
    }

    new TableFactor(varAssignment.toIndexedSeq, accValue)
  }
}





