package edu.uulm.scbayes.logging

import collection.mutable.{SynchronizedBuffer, ArrayBuffer}
import edu.uulm.scbayes.util._
import collection.Seq

trait Publisher[A] {
  private val subscribers = new ArrayBuffer[A => Unit] with SynchronizedBuffer[A => Unit]
  protected def publish(msg: A) {subscribers.foreach(_.apply(msg))}
  def subscribe(subscriber: A => Unit) {subscribers.append(subscriber)}
}

trait SeriesData[A] extends Publisher[Seq[(Int,A)]]{
  def title: String
  def data: Seq[A]
  def dataIndexed: Seq[(Int,A)]
}

class SeriesDataBuilder[A](override val title: String, val startIndex: Int = 0) extends SeriesData[A] {
  protected val _data = new ArrayBuffer[A] with SynchronizedBuffer[A]

  def append(newItem: A*) {
    var oldSize = 0
    synchronized {
      oldSize = _data.size
      _data.append(newItem:_*)
    }
    //add index to new items
    val indexedNewItems: Seq[(Int, A)] = newItem.zipWithIndex.map(t => (t._2 + oldSize + startIndex, t._1))
    publish(indexedNewItems)
  }

  def data: Seq[A] = synchronized{ _data.clone }
  def dataIndexed: Seq[(Int,A)] = synchronized {_data.zipWithIndex.map(t => (t._2 + startIndex, t._1))}
  def csv: String = {
    "idx\t%s\n%s".format(
      title,
      dataIndexed.map{case (idx, d) => "%d\t%f".format(idx,d)}.mkString("\n")
    )
  }
}

/**
 * A series of data items, where each data item consists of several data elements of the same kind.
 * There is no association of elements between steps. Thus considering the data (1,2,3) followed by
 * (7,8,9), 1 is not associated with 7 and 2 not with 8. The data (1,3,2), (7,8,9) is considered to be equal.
 */
class SampleGroup[A: Numeric](title: String, startIndex: Int = 0) extends SeriesDataBuilder[Seq[A]](title, startIndex) {
  private val meanSeries = new SeriesDataBuilder[Double]("%s (mean)".format(title), startIndex)
  private val deviationSeries = new SeriesDataBuilder[Double]("%s (deviation)".format(title), startIndex)

  this.subscribe{ newData: Seq[(Int, Seq[A])] =>
    val indexDropped: Seq[Seq[A]] = newData.map(_._2)
    meanSeries.append(indexDropped.map(_.mean):_*)
    deviationSeries.append(indexDropped.map(s => math.sqrt(s.variance)):_*)
  }

  def mean: SeriesData[Double] = meanSeries
  def deviation: SeriesData[Double] = deviationSeries
  override def csv: String = {
    "#%s\nidx\tmean\tdeviation\n%s".format(
      title,
      meanSeries.dataIndexed.zip(deviationSeries.data)
        .map{case ((idx, mean), sd) => "%d\t%f\t%f".format(idx,mean,sd)}.mkString("\n")
    )
  }

}
