package edu.uulm.scbayes.logic

import collection.mutable.WeakHashMap

/** This are the predicate or function names that can be used. */
trait AtomDefinition {
  def name: String

  def baseSignature: Seq[Sort]

  def signature: Seq[Sort]

  def isDynamic = baseSignature.headOption.map(_.isTimeSort).getOrElse(false)
  def toParseableString: String
}

trait PredicateAtomDefinition extends AtomDefinition {
  override def signature = baseSignature

  def toParseableString: String = "%s(%s)".format(name,signature.map(_.name).mkString(","))
}

trait FunctionalAtomDefinition extends AtomDefinition {
  override def baseSignature: Seq[Sort] = signature.view(0, signature.size - 1)

  def targetSignature: Sort

  def toParseableString: String = "%s(%s!)".format(name,(baseSignature :+ targetSignature).map(_.name).mkString(","))
}

sealed trait AtomBase {
  type RangeType <: Any

  def name: AtomDefinition

  def arguments: Seq[Constant]

  override def toString = "%s(%s)".format(name.name, arguments.mkString(","))

  def getTime: Option[Int] = arguments.headOption.filter(_.sort.isTimeSort).map(_.name.toInt)
}

object AtomBase {
  type ABT[T] = AtomBase { type RangeType = T }

  def apply(definition: AtomDefinition, args: Seq[Constant]) = {
    assert(definition.baseSignature == args.map(_.sort))
    definition match {
      case f: FunctionalAtomDefinition => FunctionalAtomBase(f, args.toIndexedSeq)
      case p: PredicateAtomDefinition => PredicateAtomBase(p, args.toIndexedSeq)
    }
  }
}

case class PredicateAtomBase(name: PredicateAtomDefinition, arguments: IndexedSeq[Constant]) extends AtomBase {
  type RangeType = Boolean

  override def equals(that: Any): Boolean = that match {
    case pab@PredicateAtomBase(n,a) => this.eq(pab) || (this.hashCode == that.hashCode && name == n && arguments == a)
    case _ => false
  }

  val hashCodeMemo = name.hashCode ^ arguments.hashCode
  override def hashCode = hashCodeMemo
}

final class FunctionalAtomBase private (val name: FunctionalAtomDefinition, val arguments: IndexedSeq[Constant]) extends AtomBase {
  type RangeType = Constant

  val hashMemo = name.hashCode ^ arguments.hashCode

  override def hashCode = hashMemo

//  override def equals(that: Any): Boolean = that match {
//    case fab @ FunctionalAtomBase(n2, a2) => this.eq(fab) || (this.hashCode == fab.hashCode && this.name == fab.name && this.arguments == fab.arguments)
//    case _ => false
//  }
  override def equals(that: Any): Boolean = that match {
    case fab @ FunctionalAtomBase(n2, a2) => this.name == fab.name && this.arguments == fab.arguments
    case _ => false
  }

  def copy(_name: FunctionalAtomDefinition = name, _arguments: IndexedSeq[Constant] = arguments) = FunctionalAtomBase(_name,_arguments)
}

object FunctionalAtomBase {
  private val objectStore = new WeakHashMap[(FunctionalAtomDefinition, Seq[Constant]), FunctionalAtomBase]
//  def apply(name: FunctionalAtomDefinition, arguments: IndexedSeq[Constant]) =
//    objectStore.getOrElseUpdate((name,arguments), new FunctionalAtomBase(name,arguments))
  def apply(name: FunctionalAtomDefinition, arguments: IndexedSeq[Constant]) = new FunctionalAtomBase(name,arguments)
  def fromFunctionalPredicate(f: FunctionalGroundPredicate) = {
    FunctionalAtomBase(f.predicate, f.arguments.toIndexedSeq)
  }
  def unapply(fab: FunctionalAtomBase) = Some((fab.name,fab.arguments))
}

/**
  * A random variable, that can be evaluated to true or false.
  */
sealed trait Atom extends GAtom {
  type BaseType <: AtomBase

  def base: BaseType

  def target: BaseType#RangeType

  //TODO: hacks are getting annoying
  def toPredicate: Predicate

  def predicate: AtomDefinition = base.name

  def parameters: Seq[Term] = toPredicate.parameters
}

//todo get rid of the target member and replace by a constant true
case class PredicateAtom(base: PredicateAtomBase, target: Boolean) extends Atom {
  type BaseType = PredicateAtomBase

  override def toString = base.toString
  val memohashCode: Int = base.hashCode ^ target.hashCode
  override def hashCode = memohashCode

  def toPredicate: Predicate = GroundNormalPredicate(base.name.asInstanceOf[NormalPredicateDefinition],base.arguments)
}

case class FunctionalAtom(base: FunctionalAtomBase, target: Constant) extends Atom {
  type BaseType = FunctionalAtomBase

  override def toString = "%s(%s)".format(base.name.name, (base.arguments :+ target).mkString(","))
  val memohashCode: Int = base.hashCode ^ target.hashCode
  override def hashCode = memohashCode

  def toPredicate: Predicate = new FunctionalGroundPredicate(base.name.asInstanceOf[FunctionalPredicateDefinition],base.arguments :+ target)
}

object FunctionalAtom {
  def fromFunctionalPredicate(f: FunctionalGroundPredicate) = {
    FunctionalAtom(FunctionalAtomBase.fromFunctionalPredicate(f), f.result)
  }

  def fromConstants(definition: FunctionalAtomDefinition, arguments: Seq[Constant]) = {
    require(arguments.size > 0,"trying to build a FunctionalAtom with empty argument list; must contain at least result constant")
    FunctionalAtom(FunctionalAtomBase(definition, arguments.init.toIndexedSeq), arguments.last)
  }
}