package edu.uulm.scbayes.logic

trait AtomBuilder {

  /** This method allows the construction of an atom from strings containing the name of the predicate
    * and the constants.
    *
    * @param name The name of the predicate.
    * @param arguments A list of strings that are the names of the constants. We can't take Constants here, because
    *  the sorts are unknown up till now.
    *
    * @return A (ground) atom and new constants that had to be created for it.
    */
  final def buildAtom(name: String, arguments: Seq[String]): (Atom,Seq[Constant]) = {

    val predicateDefinition = predicates.find(_.name == name).getOrElse(throw new RuntimeException("could not find predicate '%s'".format(name)))

    val sortedNames: Seq[(String, Sort)] = arguments.zip(predicateDefinition.signature)

    val (argConstants,newConstants) = lookupOrCreateConstants(sortedNames.toList)

    val atom = predicateDefinition match {
      case pad: PredicateAtomDefinition => PredicateAtom(PredicateAtomBase(pad, argConstants.toIndexedSeq),true)
      case fad: FunctionalAtomDefinition => FunctionalAtom.fromConstants(fad,argConstants)
    }
    (atom,newConstants)
  }

  //Lookup or create new constants for the given String,Sort pairs.
  //First of result is new signature, second are the newly created constants.
  def lookupOrCreateConstants(sn: List[(String,Sort)],
                              result: List[Constant] = Nil,
                              newConstants: List[Constant] = Nil): (Seq[Constant],Seq[Constant]) = sn match {
    case Nil => (result.reverse, newConstants.distinct) //only first needs to be in correct order
    case (name,sort) :: tail => lookupConstantTyped(name,sort)
      .map(foundConstant => lookupOrCreateConstants(tail,foundConstant :: result, newConstants))
      .getOrElse{
      val newConstant = Constant(name,sort)
      lookupOrCreateConstants(tail,newConstant :: result, newConstant :: newConstants)}
  }

  final def lookupConstantTyped(name: String, sort: Sort): Option[Constant] = constantsBySort(sort).find(_.name == name)

  final def createPredicateInstanceFixSorts(pname: String, args: List[Term]): Either[String,Predicate] = {
    def zipOpt[A,B](as: Seq[A], bs: Seq[B]) = if(as.size == bs.size) Some(as zip bs) else None
    //try to fix the sorts of the constants
    for(
      p <- getPredicateByName(pname).right;
      zipped <- zipOpt(args,p.parameters).toRight("number of parameters does not fit").right
    ) yield Predicate(p,zipped.map(_._1))
  }

  def getPredicateByName(pname: String): Either[String,AbstractPredicateDefinition] =
    abstractPredDefs.find(_.name == pname).toRight("predicate %s undefined".format(pname))

  def constantsBySort: Map[Sort, Set[Constant]]
  def predicates: Iterable[AtomDefinition]
  def abstractPredDefs: Set[AbstractPredicateDefinition]
}







