package edu.uulm.scbayes.logic

import java.lang.String

//here starts the formula stuff
sealed trait Formula {
  implicit def formula2formulaOps(f: Formula) = new FormulaOps(f)
}

case class Existential(boundVar: Variable, child: Formula) extends Formula{
  override def toString: String = "EXISTS %s %s".format(boundVar, child)
}

case class ForAll(boundVar: Variable, child: Formula) extends Formula{
  override def toString: String = "FORALL %s %s".format(boundVar, child)
}

case class Conjunction(children: Set[Formula]) extends Formula {
  override def toString = if(children.size == 1)
    children.head.toString
  else
    "(%s)".format(children.map(_.toString).mkString(" ^ "))
}

case class Disjunction(children: Set[Formula]) extends Formula {
  override def toString = if(children.size == 1)
    children.head.toString
  else
    "(%s)".format(children.map(_.toString).mkString(" v "))
}

case class Negation(child: Formula) extends Formula{
  override def toString = child match {
    case ga: GAtom => "!%s".format(child)
    case _ => "!(%s)".format(child)
  }
}

trait GAtom extends Formula {
  def predicate: AtomDefinition
  def parameters: Seq[Term]
}

object GAtom{
  def unapply(ga: GAtom) = (ga.predicate, ga.parameters)
}
// predicate stuff ****

/**
 * A predicate instance that can have both constants and variables as arguments.
 * Create instances using the companion object's apply. This way, you'll obtain an instance
 * of GroundPredicate automatically if all terms are constants.
 */
class Predicate protected (val predicate: AtomDefinition, val parameters: Seq[Term])
  extends GAtom with Product2[AtomDefinition, Seq[Term]]{

  assert(predicate.signature.size == parameters.size)

  def _1: AtomDefinition = predicate
  def _2: Seq[Term] = parameters
  def canEqual(that: Any): Boolean = that match {
    case p: Predicate => true
    case _ => false
  }

  override def equals(x: Any): Boolean = x match {
    case that@Predicate(p2,par2) => this.hashCode == that.hashCode && p2 == predicate && par2 == parameters
    case _ => false
  }
  val hashMemo = predicate.hashCode ^ parameters.hashCode
  override def hashCode = hashMemo
  override def toString = predicate.name + "(" + parameters.mkString(",") + ")"
}

object Predicate{
  def apply(pd: AtomDefinition, pars: Seq[Term]) = {
    val constants = pars.collect{case c: Constant => c}
    if(pars.forall(_.isInstanceOf[Constant]))
      GroundPredicate(pd,constants)
    else
      new Predicate(pd,pars)
  }
  def unapply(pred: Predicate) = Some((pred.predicate,pred.parameters))
}

