package edu.uulm.scbayes.logic

import java.lang.String
import scalaz.Monoid
import edu.uulm.scbayes.util._

trait ConstantDomain {
  def constants: Map[Sort, Set[Constant]]
}

trait LogicSignature {
  def sorts: Set[Sort]
  def predicates: Set[AtomDefinition]
}

/**
 * Holds information about a first-order signature.
 *
 * The constructor requires, that the `constants` argument is respecting the isA relations. Also
 * the isA map must include the transitive relations. This is automatically computed when using the
 * factory method of the companion object.
 *
 *
 * Date: 3/21/11
 */

class Signature private[Signature](val sorts: Set[Sort],
                                   val predicates: Set[AtomDefinition],
                                   val constants: Map[Sort,Set[Constant]],
                                   val isARelations: Map[Sort,Set[Sort]] = Map[Sort,Set[Sort]]().withDefaultValue(Set[Sort]()))
  extends ConstantDomain with AtomBuilder {

  require(
    (for(p <- predicates; sort <- p.baseSignature) yield sort)
      .toList
      .distinct
      .forall(s => sorts.contains(s)),
    "not all sorts used as parameter to a predicate are supplied")

  require(sorts.forall(s => !isARelations(s).contains(s)), "no cycles in the sort hierarchy allowed (or recommended:)")

  /**
  * @return A seq of all possible ground predicates.
  */
  def herbrandBase: Set[AtomBase] = for {
    pred <- predicates
    ranges = pred
      .baseSignature
      .map(sort => constants(sort).toSeq)
    ground_params <-  crossProduct(ranges).iterator
  } yield AtomBase(pred, ground_params)

  /**
  * @return a seq of all possible normal predicates (no functional predicates). */
  def normalBase: Set[PredicateAtomBase] = {
    val normalPredicates = predicates
      .collect{case pad: PredicateAtomDefinition => pad}

    for(
      npd <- normalPredicates;
      args <- crossProduct(npd.baseSignature.map(constants(_).toSeq)).iterator.toSeq
    ) yield PredicateAtomBase(npd, args.toIndexedSeq)
  }

  /**
  * @return a seq of all possible function bases.
  */
  def functionalBase: Set[FunctionalAtomBase] = {
    val functionalPredicates = predicates
      .collect{ case fpd: FunctionalAtomDefinition => fpd}

    for(
      fpd <- functionalPredicates;
      args <- crossProduct(fpd.baseSignature.map(constants(_).toSeq)).iterator.toSeq
    ) yield FunctionalAtomBase(fpd,args.toIndexedSeq)
  }

  def interpretationsEqual(i1: TruthAssignment, i2: TruthAssignment): Boolean =
  {
    this.normalBase.forall( gp => i1.truth(gp) == i2.truth(gp))
    this.functionalBase.forall(fb => i1.functionValue(fb) == i2.functionValue(fb))
  }

  /** Finds a Constant by name. */
  def lookupConstant(name: String, sort: Sort): Option[Constant] = constants(sort).find(_.name == name)

  def buildFunctionBase(predicate: String, arguments: String*): AtomBase = {

    def findArguments(fd: Seq[Sort]): IndexedSeq[Constant] = {
      fd
        .zip(arguments)
        .map{ case (sort, constant) => constants(sort).find(_.name == constant).get }
        .toIndexedSeq
    }

    val predicateDefinition = predicates.find(_.name == predicate).get
    predicateDefinition match {
      case fd: FunctionalPredicateDefinition => FunctionalAtomBase(fd, findArguments(fd.arguments))
      case pd: NormalPredicateDefinition => PredicateAtomBase(pd, findArguments(pd.signature))
    }
  }

  def setConstants(newConstants: Map[Sort,Set[Constant]]): Signature = Signature(
    sorts,
    predicates,
    constants ++ newConstants,
    isARelations
  )

  //stuff to implement AtomBuilder
  def constantsBySort: Map[Sort, Set[Constant]] = constants

  def abstractPredDefs: Set[AbstractPredicateDefinition] = predicates.map(_.asInstanceOf[AbstractPredicateDefinition])

  override def toString: String =
    "Signature(sorts=(%s),predicates=(%s),constants=(%s),subsorts=(%s)".format(
      sorts.map(_.name).mkString(","),
      predicates.mkString(","),
      sorts.map(s => "%s = {%s}".format(s.name,constants(s).mkString(","))).mkString(","),
      isARelations
    )

  /** Retrieve only the constants that are of the given sort without yielding any constants of subsorts. */
  def constantsByPrimitiveSort(sort: Sort): Set[Constant] = constants(sort).filter(_.sort == sort)

  def toParseableString: String = {
    val predicateDefinitions = predicates
      .map(_.toParseableString)
      .mkString("\n")
    val isAs = isARelations
      .flatMap{case (sort, superSorts) => superSorts.map("%s isA %s".format(sort, _))}
      .mkString("\n")
    val constantDomains = sorts
      .map(sort => "%s = {%s}".format(sort.name, constantsByPrimitiveSort(sort).toList.sortBy(_.toString).mkString(",")))
      .mkString("\n")

    "//predicates\n%s".format(predicateDefinitions) +
      (if(isAs.isEmpty) "" else "\n\n//isA relations\n%s".format(isAs)) +
      "\n\n//constant domains\n%s".format(constantDomains)
  }

  def addConstants(newConstants: Iterable[Constant]): Signature = {
    val grouped: Map[Sort, Set[Constant]] = newConstants.toSet.groupBy(_.sort)
    val mergedWithOldConstants = constants ++ grouped.map{case (s,cs) => s -> (constants(s) ++ cs)}
    setConstants(mergedWithOldConstants)
  }
  def addPredicates(pd: AtomDefinition*) = Signature(sorts, predicates ++ pd, constants, isARelations)
  def addSorts(ss: Sort*) = Signature(sorts ++ ss, predicates, constants, isARelations)
  def addIsARelations(isAs: (Sort,Set[Sort])*): Signature = Signature(
    sorts,
    predicates,
    constants,
    (isARelations.toSeq ++ isAs).groupBy(_._1).mapValues(_.flatMap(_._2).toSet))
}

object Signature {
  def buildTransitiveConstantMap(isARelations: Map[Sort, Set[Sort]], constants: Map[Sort, Set[Constant]], sorts: Set[Sort]): Map[Sort, Set[Constant]] = {
    val transitiveIsA: Map[Sort, Set[Sort]] = transitiveClosure(isARelations)
    val isSubsumedByA = reverseMultiMap(transitiveIsA).withDefaultValue(Set[Sort]())

    def allConstantsOfSort(s: Sort): Set[Constant] = for (
      ss <- isSubsumedByA(s) + s;
      c <- constants.withDefaultValue(Set[Constant]())(ss)
    ) yield c

    val transitiveConstants = for(
      s <- sorts
    ) yield s -> allConstantsOfSort(s)

    transitiveConstants.toMap
  }

  def apply(sorts: Set[Sort],
            predicates: Set[AtomDefinition],
            constants: Map[Sort,Set[Constant]],
            isARelations: Map[Sort,Set[Sort]] = Map[Sort,Set[Sort]]().withDefaultValue(Set[Sort]())): Signature = {

    //compute the transitive isA relations
    val transitiveConstants = buildTransitiveConstantMap(isARelations, constants, sorts)

    new Signature(sorts, predicates, transitiveConstants, transitiveClosure(isARelations).withDefaultValue(Set[Sort]()))
  }

  def empty: Signature = Signature(Set(),Set(),Map.empty)

  implicit val signatureIsMonoid: Monoid[Signature] = new Monoid[Signature]{
    def append(s1: Signature, s2: => Signature): Signature =
      s1
        .addSorts(s2.sorts.toSeq:_*)
        .addPredicates(s2.predicates.toSeq:_*)
        .addConstants(s2.constants.values.flatten)
        .addIsARelations(s2.isARelations.toSeq:_*)

    val zero: Signature = Signature.empty
  }
}



