package edu.uulm.scbayes.logic.cnf

import edu.uulm.scbayes.logic._


/**
  * Operations that can be performed on a GroundCNF instance.
  *
  *
  * Date: 03.05.11 */
object CNFOps {
  def reduceWithEvidence(cnf: GroundCNF, evidence: TruthAssignment): GroundCNF = {
    /** The reduced clause or None if the clause becomes true under evidence.
      */
    def reduceClause(clause: GroundClause): Option[GroundClause] = {
      val satisfyPositive = clause.plainAtoms.exists(evidence.lift(_) == Some(true))
      val satisfyNegated = clause.negatedAtoms.exists(evidence.lift(_) == Some(false))
      if (satisfyPositive || satisfyNegated)
        None //the clause becomes true
      else
        Some(Clause(
          clause.plainAtoms.filter(!evidence.isDefinedAt(_)),
          clause.negatedAtoms.filter(!evidence.isDefinedAt(_))
        ))
    }

    val newClauses = cnf.clauses.map(reduceClause).flatten: Seq[GroundClause]

    //if a clause became empty, it is false and the whole cnf is not satisfiable
    val result = GroundCNF.fromClauses(newClauses)
//    if (newClauses.exists(cl => cl.plainAtoms.isEmpty && cl.negatedAtoms.isEmpty))
//      GroundCNF.fromClauses(Seq(FalseClause))
//    else {
//      GroundCNF.fromClauses(newClauses)
//    }

    assert(
      result.clauses.forall(cl => cl.atoms.forall(!evidence.isDefinedAt(_))),
      "reduced cnf may not contain any evidence atom"
    )
    result
  }

  /** Try to figure out certain assignments and return them if any have been found.
    *
    * Easy victims are unit clauses. */
  def findEvidence(cnf: GroundCNF): Option[TruthAssignment] = {
    val positiveUnitClauses = cnf.clauses.filter(cl => cl.plainAtoms.size == 1 && cl.negatedAtoms.size == 0)
    val negativeUnitClauses = cnf.clauses.filter(cl => cl.plainAtoms.size == 0 && cl.negatedAtoms.size == 1)

    val positiveAtoms: Set[Atom] = positiveUnitClauses.map(_.plainAtoms.head).toSet
    val negativeAtoms: Set[Atom] = negativeUnitClauses
      .map(_.negatedAtoms.head)
      //we can't treat negative functional atoms
      .collect {
      case pa: PredicateAtom => pa
    }
      .toSet

    val positivePredicateMap = positiveAtoms.collect {
      case p: PredicateAtom => p.base -> true
    }
    val negativePredicateMap = negativeAtoms.collect {
      case p: PredicateAtom => p.base -> false
    }
    val positiveFunctionMap = positiveAtoms.collect {
      case f: FunctionalAtom => f.base -> f.target
    }

    if (Seq(positivePredicateMap, negativePredicateMap, positiveFunctionMap).exists(!_.isEmpty))
    //if we have found something we return a TruthAssignment
      Some(TruthAssignment((positivePredicateMap ++ negativePredicateMap).toMap, positiveFunctionMap.toMap))
    else
    //this signals failure of finding easy assignments
      None
  }

  def findInequalities(cnf: GroundCNF): Map[FunctionalAtomBase,Set[Constant]] = {
    val negatedFunctionalAtoms = for(
      clause <- cnf.clauses if(clause.plainAtoms.size == 0 && clause.negatedAtoms.size == 1);
      inequality <- clause.negatedAtoms.collectFirst{case fa: FunctionalAtom => fa}
    ) yield inequality

    negatedFunctionalAtoms.map{case FunctionalAtom(base,target) => base -> target}.groupBy(_._1).mapValues(_.unzip._2.toSet)
  }

  def histogram(cnf: GroundCNF) = cnf.clauses.groupBy(_.atoms.size).mapValues(_.size).toList.sortBy(_._1)

  def propagate(cnf: GroundCNF): (GroundCNF, TruthAssignment) = {
    //construct a sequence of simplifications
    val iter: Iterator[Option[(GroundCNF, TruthAssignment)]] =
      Iterator.iterate[Option[(GroundCNF, TruthAssignment)]](Some(cnf, TruthAssignment.empty)) {
        case Some((tcnf, tta)) => findEvidence(tcnf).map(rta => (reduceWithEvidence(tcnf, rta), rta))
        case None => None
      }


    iter.takeWhile(_.isDefined)                     //only take real simplifications
      .map(_.get)                                   //unpack the options
      .reduceLeft[(GroundCNF, TruthAssignment)]{    //get the last formula and compose the TruthAssignments
      case ((cnf1, ta1), (cnf2, ta2)) => (cnf2, ta1 orElse ta2)
    }
  }
}