package edu.uulm.scbayes.logic.cnf

import edu.uulm.scbayes.logic._
import collection.immutable.Set

/**
 * Trait that can be added to Conjunctions to mark them as being in conjunctive normal form.
 *
 * Date: 29.03.11
 */

trait ConjunctiveNormalForm extends Conjunction{
  self: Conjunction =>

  val clauses: Seq[Disjunction] = {
    children.toSeq.map {
      case dj: Disjunction =>
        assert(
          dj.children.forall {
            case p: GAtom => true
            case Negation(p: GAtom) => true
            case _ => false
          },
          "all children of conjunction must be disjunctions in a CNF (%s)".format(this.toString))
        dj
      case p: GAtom => Disjunction(Set(p))
      case Negation(p: GAtom) => Disjunction(Set(Negation(p)))
      case _ => throw new RuntimeException("encountered invalid clause for CNF")
    }
  }

  def isTautology = clauses.forall(FormulaOps.isTautology)
}


object ConjunctiveNormalForm {
  def fromConjunction(cn: Conjunction) = {
    //wrap single atom children into disjunctions
    val children: Set[Formula] = cn.children.map{
      case d: Disjunction => d
      case n@Negation(ga: GAtom) => Disjunction(Set(n))
      case ga: GAtom => Disjunction(Set(ga))
      case _  => throw new RuntimeException("cannot build cnf from %s".format(cn))
    }
    new Conjunction(children) with ConjunctiveNormalForm
  }
}
