package edu.uulm.scbayes.logic.cnf

import edu.uulm.scbayes.logic._

/** Representation of totally ground CNF that is independent of Formula hierarchy.
  *
  *
  * Date: 29.03.11 */

class GroundCNF(val clauses: IndexedSeq[GroundClause]) {
  def toFormula = Conjunction(clauses.map(cl => Disjunction(cl.plainAtoms.map(_.toPredicate) ++ cl.negatedAtoms.map(a => Negation(a.toPredicate)) toSet)).toSet)
  def invert: GroundCNF = GroundCNF.formulaToGroundCNF(Negation(toFormula))

  @deprecated("maybe you want to use atom bases instead?")
  lazy val atoms: Seq[Atom] = clauses.flatMap(gc => gc.plainAtoms ++ gc.negatedAtoms).distinct
  lazy val atomBases = clauses.flatMap(_.atomBases).distinct

  def evaluate(assignment: Interpretation): Boolean = clauses.forall{clause =>
    val positiveSatisfaction = clause.plainAtoms.exists(assignment)
    val negativeSatisfaction = clause.negatedAtoms.exists(a => !assignment(a))
    positiveSatisfaction || negativeSatisfaction
  }

  /** If this formula evaluates to a constant value independent of any variable assignment,
    * this method returns this value. Otherwise it returns None. */
  lazy val unconditionalValue: Option[Boolean] = {
    if(clauses.forall(_.unconditionalValue == Some(true)))
      Some(true)
    else if(clauses.exists(_.unconditionalValue == Some(false)))
      Some(false)
    else
      None
  }

  override def toString: String = clauses.map(_.toString).mkString(" ^ ")

  override def equals(that: Any): Boolean = that match {
    case gcnf: GroundCNF => this.hashCode == that.hashCode && this.clauses == gcnf.clauses
    case _ => false
  }

  val memHashCode = clauses.hashCode
  override def hashCode: Int = memHashCode
}

object GroundCNF {
  def disjunction2Clause(dj: Disjunction): GroundClause = {

    val onlyAtoms = dj.children.forall(atom => atom match {
      case a: GroundPredicate => true
      case a: Atom => true
      case Negation(a: GroundPredicate) => true
      case Negation(a: Atom) => true
      case _ => false
    })

    assert(onlyAtoms, "GroundClause may only contain ground atoms (Atom or GroundPredicate")

    Clause(
      dj.children.collect{
        case a: Atom => a
        case a: GroundPredicate => predicate2Atom(a)
      }.toArray,
      dj.children.collect{
        case Negation(a: Atom) => a
        case Negation(a: GroundPredicate) => predicate2Atom(a)
      }.toArray
    )
  }

  def fromConjunction(cn: Conjunction) = new GroundCNF( cn.children.map{
      case dj: Disjunction => disjunction2Clause(dj)
      case pred: GAtom => disjunction2Clause(Disjunction(Set(pred)))
      case Negation(pred: GAtom) => disjunction2Clause(Disjunction(Set(Negation(pred))))
      case _ => throw new RuntimeException("conversion to cnf: conjunction contains invalid element")
    }(collection.breakOut))

  def fromClauses(_clauses: Iterable[GroundClause]) = new GroundCNF(_clauses.toIndexedSeq)

  def predicate2Atom(p: GroundPredicate) = {
    p match {
      case p: GroundNormalPredicate => PredicateAtom(PredicateAtomBase(p.name,p.parameters.toIndexedSeq),true)
      case f: FunctionalGroundPredicate => FunctionalAtom.fromFunctionalPredicate(f)
    }
  }

  /**
   * Returns a CNF-version of the formula.
   * For this to work, the formula must be free of variables.
   */
  def formulaToCNF(formula: Formula): ConjunctiveNormalForm = {
    /*
    1. push negations to predicates
    2. surround with a conjunction
    3. simplify (merge nested conjunctions, disjunctions)
    4. for any nested disjunction that contains a conjunction, apply distributive law; if done so, goto 3
     */

//    require(!formula.extractTerms.exists(_.isInstanceOf[Variable]), "must give me ground formula")

    var f = Conjunction(Set(formula.pushNegations))
    var finished = true
    do {
      finished = true
      f = Conjunction(f.children.map(_.simplify))
      f = FormulaOps.flatten(f)

      val disjunctions = f.children.collect{case dj: Disjunction => dj}
      val non_disjunctions = f.children -- disjunctions
      val unfinished_disjunctions = disjunctions.filter(dj => dj.children.exists(_.isInstanceOf[Conjunction]))
      val finished_disjunctions = disjunctions -- unfinished_disjunctions

      //apply distributive law to every unfinished clause and signal we need another loop
      if(!unfinished_disjunctions.isEmpty) {
        finished = false
        f = Conjunction(
          non_disjunctions ++
            finished_disjunctions ++
            unfinished_disjunctions.map(FormulaOps.distribute)
        )
      }
    } while(!finished)

    //throw away tautological clauses
    val result = Conjunction(f.children.filter{
      case dj: Disjunction => !FormulaOps.isTautology(dj)
      case _ => true
    })
    ConjunctiveNormalForm.fromConjunction(result)
  }

  def formulaToGroundCNF(formula: Formula): GroundCNF = {
    assert(formula.extract.collect{case pred: GAtom => pred}.forall(x => x.isInstanceOf[GroundPredicate] || x.isInstanceOf[Atom]))
    val cnf = GroundCNF.formulaToCNF(formula)
    GroundCNF.fromConjunction(cnf)
  }
}