package edu.uulm.scbayes.logic.cnf

import edu.uulm.scbayes.logic._


/**
 * GroundClause constitutes the main component of a GroundCNF.
  *
 *
 * Date: 29.03.11
 */

trait GroundClause {
  def plainAtoms: Array[Atom]

  def negatedAtoms: Array[Atom]
  val atomBases: Seq[AtomBase] = (plainAtoms ++ negatedAtoms map (a => a.base)).distinct
  def isUnit = (plainAtoms.size + negatedAtoms.size) == 1
  def evaluate(interp: Interpretation): Boolean = plainAtoms.exists(interp(_)) || negatedAtoms.exists(!interp(_))
  def unsatisfiedAtoms(interp: Interpretation): Array[Atom] =
    (plainAtoms.filterNot(interp) ++ negatedAtoms.filter(interp)).distinct
  def contains(a: Atom) = plainAtoms.contains(a) || negatedAtoms.contains(a)
  def atoms: Array[Atom] = (plainAtoms ++ negatedAtoms).distinct
  def unconditionalValue: Option[Boolean] = {
    if(atoms.isEmpty)
      Some(false)
    else if (!plainAtoms.intersect(negatedAtoms).isEmpty)
      Some(true)
    else
      None
  }
  override def toString: String = (plainAtoms.map(_.toString) ++ negatedAtoms.map("!%s".format(_))).mkString(" v ")
}

case class Clause(plainAtoms: Array[Atom], negatedAtoms: Array[Atom]) extends GroundClause {
  override def equals(that: Any): Boolean = that match {
    case cl@Clause(pa, na) => cl.hashCode == this.hashCode && this.plainAtoms.toSeq == pa.toSeq && this.negatedAtoms.toSeq == na.toSeq
    case _ => false
  }

  val memHash: Int = plainAtoms.toSeq.hashCode ^ negatedAtoms.toSeq.hashCode

  override def hashCode: Int = memHash
}

object TrueClause extends GroundClause {
  override val plainAtoms = Array[Atom]()
  override val negatedAtoms = Array[Atom]()
  override def toString = "TRUE"
  override val unconditionalValue = Some(true)
}

object FalseClause extends GroundClause {
  override val plainAtoms = Array[Atom]()
  override val negatedAtoms = Array[Atom]()
  override def toString = "FALSE"
  override val unconditionalValue = Some(false)
}