package edu.uulm.scbayes.logic.cnf.dimacs

import edu.uulm.scbayes.logic._
import edu.uulm.scbayes.logic.cnf._

/**
  * Reads a SAT instance from a text file in the cnf format of the DIMACS format.
  *
  * The format is specified in the following file:
  * http://www.cs.ubc.ca/~hoos/SATLIB/Benchmarks/SAT/satformat.ps
  *
  *
  * Date: 02.05.11 */

case class DIMACSInstance(description: String, numVariables: Int, clauses: Seq[DimacsClause]) {
  val sort: Sort = Sort("CNF-Atom")

  val constants: IndexedSeq[Constant] =
    for( i <- 1 to numVariables ) yield Constant(i.toString, sort)

  val predicateAtomDefinition = new PredicateAtomDefinition {
    def baseSignature: Seq[Sort] = Seq(sort)

    def name: String = "p"
  }

  val atoms: IndexedSeq[PredicateAtom] =
    constants.map(c => PredicateAtom(PredicateAtomBase(predicateAtomDefinition,Array(c)), true))

  def asGroundCNF: GroundCNF = {
    val groundClauses = clauses.map(cl => Clause(cl.plainVariables.map(int2Atom).toArray, cl.negatedVariables.map(int2Atom).toArray))
    GroundCNF.fromClauses(groundClauses)
  }

  def int2Atom(i: Int): Atom = atoms(i - 1)

  def constantDomain: ConstantDomain = new ConstantDomain {
    val constants = Map(sort -> DIMACSInstance.this.constants.toSet)
  }
}



