package edu.uulm.scbayes.logic.cnf.sampling

import util.Random
import edu.uulm.scbayes.util._

/**
 * Implementation of SampleSAT algorithm described in [1].
 *
 * This implementation is extended to multivalued atoms.
 *
 * [1] Wei, W.; Erenrich, J. & Selman, B. Towards efficient sampling: Exploiting random walk strategies
 *
 * @tparam C type of clause
 * @tparam A type of atom
 *
 *
 * Date: 25.03.11
 */
trait MutableSATWalker[TCl,TC] extends MutableCNFAssignment {

  def isClauseSatisfied(clause: TCl): Boolean
  /** @return All the changes that would make the given clause true. */
  def satisfyingChanges(unsatClause: TCl): Seq[TC]
  def clauses: Seq[TCl]
  def applyChange(change: TC)
  /** @return A random change of the given atom. */
  def createRandomFlip(random: Random): TC

  /**Difference of satisfied clauses before and after change.
   * Positive means the change increases the number of satisfied clauses. */
  def score(change: TC): Int

  def unsatisfiedClauses: Seq[TCl] = clauses.filterNot(isClauseSatisfied)

  /** Performs one WalkSAT step.
   * Satisfy a random unsatisfied clause by flipping one atom such that as many other clauses as possible become true.*/
  def walkSAT(random: Random) {
    val unsatClauses = unsatisfiedClauses

    //if the problem is already satisfied, we cannot do anything
    if(unsatClauses.isEmpty){
      assert(isSatisfied)
      return
    }

    val clause = unsatClauses.pickRandom(random)
    val candidates = satisfyingChanges(clause)
    val bestChanges = maxByMultiple(candidates)(score)
    val bestChange = bestChanges.pickRandom(random)

    applyChange(bestChange)
  }

  /**
  * 1. pick an atom at random
  * 2. if flipping it results in at least as many satisfied clauses as before, do it
  * 3. otherwise, flip it with probability exp(-cost/temperature), where cost is difference in satisfied clauses. */
  def simulatedAnnealing(random: Random, temperature: Double) {
    //should be pick the best flip? For now we don't.
    val change = createRandomFlip(random)
    val changeScore = score(change)

    //see if we flip the atom
    val makeFlip = changeScore >= 0 || random.nextDouble() < math.exp(changeScore / temperature)
    if(makeFlip)
      applyChange(change)
  }

  /** Perform a SampleSAT step.
   * @param probWalkSAT Probablity to perform a greedy walkSAT step.
   * @param temperatureSA the temperature to use for simulated annealing.*/
  final def sampleSAT(random: Random, probWalkSAT: Double, temperatureSA: Double) {
    if (random.nextDouble() < probWalkSAT)
      this.walkSAT(random)
    else
      simulatedAnnealing(random, temperatureSA)
  }
}