package edu.uulm.scbayes.mln.factorgraph

import edu.uulm.scbayes.logic._
import edu.uulm.scbayes.probabilities.DiscreteVariable

/**
 * A variable node for factor graphs, that is associated with a ground atom.
 *
 *
 * Date: 3/14/11
 */

sealed trait LogicNode extends DiscreteVariable {
  type AtomType <: AtomBase
  def atomBase: AtomType
  def domain2Int(dv: AtomType#RangeType): Int
  def int2Domain(di: Int): AtomType#RangeType

  override def valueName(i: Int): String = int2Domain(i).toString
}

object LogicNode {
  def apply(base: AtomBase, signature: Signature): LogicNode = {
    base match {
      case pab: PredicateAtomBase => PredicateNode(pab)
      case fab: FunctionalAtomBase => FunctionValuedNode(fab,signature.constants(fab.name.targetSignature))
    }
  }
}

case class PredicateNode(predicate: PredicateAtomBase) extends LogicNode {

  type AtomType = PredicateAtomBase

  override def toString = predicate.toString
  override def atomBase = predicate

  def domainSize: Int = 2
  def domain2Int(a: Boolean): Int = if(a) 1 else 0
  def int2Domain(v: Int): Boolean = v != 0

  //optimizations start here
  val memHashCode = predicate.hashCode
  override def hashCode = memHashCode
}

case class FunctionValuedNode(functionBase: FunctionalAtomBase, range: Set[Constant])
  extends LogicNode {

  /** Constants sorted by their name */
  val sortedRange = range.view.toSeq.sortBy(_.name)

  type AtomType = FunctionalAtomBase

  override def toString = functionBase.toString
  override def atomBase = functionBase

  def domain2Int(a: Constant): Int = sortedRange.indexOf(a)
  def int2Domain(v: Int): Constant = sortedRange(v)
  def domainSize: Int = range.size

  //optimizations start here
  override def equals(that: Any): Boolean = {
    that match {
      case fvn: FunctionValuedNode => {
        fvn.hashCode == this.hashCode && this.functionBase == fvn.functionBase //&& this.range == fvn.range
      }
      case _ => false
    }
  }
  val memHash = functionBase.hashCode ^ range.hashCode
  override def hashCode = memHash
}



















