package edu.uulm.scbayes.mln.parsing

import util.parsing.combinator.JavaTokenParsers
import edu.uulm.scbayes.logic._

import scalaz._
import Scalaz._
import collection.Iterable

/**
 * Allows parsing of propositional formulas with the following syntax for variables.
 * - boolean variables are made up of letters and underscores "[a-zA-Z\_]+"
 * - multi-valued variable identifiers are string of letters and under scores followed by an integer "[a-zA-Z\_]*[0-9]+".
 *   This integer is the number of values.
 *
 * #Translation
 * ##Boolean variables
 * The variable `a` is translated to an atom `p(a)` where `p` is the generic predicate taking a constant of type
 * `Proposition` used for all binary atoms.
 *
 * ##Multi-valued variables
 * The variable `a3=2` is translated to an atom `f_a3(2)`. The sort of the argument to `f_a3` is the sort `S3` used for
 * all variables with a domain size of three.
 *
 *
 * Date: 28.11.11
 */

object PropFormulaParser extends JavaTokenParsers {
  trait Propositional
  type PSig = Signature with Propositional

  val binVarLiteral = """[a-zA-Z\_]+""".r
  val MvLiteral = """([a-zA-Z0-9\_]*[a-zA-Z\_])([0-9]+)""".r

  val propSort = Sort("Proposition")
  val propPredicate = NormalPredicateDefinition("p", List(propSort))
  val baseSignature: PSig = Signature(
    Set(Sort("Proposition")),
    Set(propPredicate),
    Map.empty,
    Map.empty
  ).asInstanceOf[PSig]

  def getOrUpdateProposition(signature: PSig, propString: String, value: Boolean = true): (PredicateAtom, PSig) = {
    val propConstant = Constant(propString, propSort)
    val atomBase = PredicateAtomBase(propPredicate, IndexedSeq(propConstant))
    val atom = PredicateAtom(atomBase, value)
    val newSignature = signature.addConstants(Set(propConstant)).asInstanceOf[PSig]

    (atom, newSignature)
  }

  def getOrUpdateMVAtom(signature: PSig, propString: String, rangeSize: Int, value: Int): (FunctionalAtom, PSig) = {
    val sort = Sort("S" + rangeSize)
    val funDef = FunctionalPredicateDefinition("f_%s%d".format(propString,rangeSize),sort :: Nil)
    val valueConstant = Constant(value.toString, sort)
    val base = FunctionalAtomBase(funDef, IndexedSeq.empty)
    val atom = FunctionalAtom(base, valueConstant)

    val newSignature = signature
      .addSorts(sort)
      .addConstants((0 until rangeSize).map(i => Constant(i.toString,sort)))
      .addPredicates(funDef)

    (atom, newSignature.asInstanceOf[PSig])
  }

  def sigRepSep[T](parser:PSig => Parser[(T,PSig)], sep: String)(implicit sig: PSig): Parser[(List[T],PSig)] =
    opt(parser(sig)) >> {
      case None => success((Nil, sig))
      case Some((t,newSig)) => opt(sep ~> sigRepSep(parser, sep)(newSig)) ^^ {
        case None => (t :: Nil, newSig)
        case Some((ts, nnSig)) => (t :: ts, nnSig)
      }
    }

  def sigRep1Sep[T](parser:PSig => Parser[(T,PSig)], sep: String)(implicit sig: PSig): Parser[(List[T],PSig)] =
    parser(sig) >> {
      case (t,newSig) => opt(sep ~> sigRepSep(parser, sep)(newSig)) ^^ {
        case None => (t :: Nil, newSig)
        case Some((ts, nnSig)) => (t :: ts, nnSig)
      }
    }

  def binVar(implicit sig: PSig): Parser[(PredicateAtom, PSig)] = binVarLiteral ^^ (getOrUpdateProposition(sig, _))

  def mvVar(implicit sig: PSig): Parser[(FunctionalAtom, PSig)] = (MvLiteral <~ "=") ~ (wholeNumber ^^ (_.toInt)) ^^ {
    case MvLiteral(id, range) ~ value => getOrUpdateMVAtom(sig, id, range.toInt, value)
  }
  def literal(implicit signature: PSig) = opt("!") ~ (mvVar | binVar | ("(" ~> formula <~ ")")) ^^ {
    case Some("!") ~ fs => ((f: Formula) => Negation(f)) <-: fs
    case None ~ fs => fs
  }

  def conjunction(implicit sig: PSig): Parser[(Formula, PSig)] = sigRep1Sep(literal(_), "^") ^^ {
    case (single :: Nil, nSig) => (single, nSig)
    case (formulas, nSig) => (Conjunction(formulas.toSet), nSig)
  }

  def disjunction(implicit signature: PSig) = sigRep1Sep(conjunction(_), "v") ^^ {
    case (single :: Nil, nSig) => (single, nSig)
    case (formulas, nSig) => (Disjunction(formulas.toSet), nSig)
  }

  def implication(implicit signature: PSig) = disjunction <~ "=>" >> {
    case (d1,nSig) => disjunction ^^ {
      case (d2, nnSig) => (Disjunction(Set(Negation(d1), d2)), nnSig)
    }
  }

  def equivalence(implicit signature: PSig) = disjunction <~ "<=>" >> {
    case (d1,nSig) => disjunction ^^ {
      case (d2, nnSig) => (
        Disjunction(Set(
          Conjunction(Set(d1, d2)),
          Conjunction(Set(Negation(d1), Negation(d2)))
        )),
        nnSig
        )
    }
  }

  def formula(implicit signature: PSig = baseSignature): Parser[(Formula,PSig)] = (
    implication
      | equivalence
      | disjunction
    )

  def createAtoms(fs: Iterable[Formula]): Iterable[Atom] = fs.map{f => f match {
    case pa: PredicateAtom => pa
    case Negation(pa: PredicateAtom) => pa.copy(target = !pa.target)
    case fa: FunctionalAtom => fa
    case Negation(fa: FunctionalAtom) => throw new RuntimeException("can not treat negated functional atom as assignment")
    case x => throw new RuntimeException("cannot create atom from formula (%s)".format(x))
  }}

  def assignment(implicit signature: PSig = baseSignature): Parser[(TruthAssignment, PSig)] = sigRepSep(literal(_), ",") ^^ { case (atoms, newSig) =>
    (TruthAssignment.fromAtoms(createAtoms(atoms)), newSig)
  }
}