package edu.uulm.scbayes.mln.parsing

import edu.uulm.scbayes.logic._
import scala.collection.mutable

/**
 * Tracks sorts, predicates and variables during parsing.
 *
 * Date: 3/11/11
 */

class SignatureBuilder extends AtomBuilder {
  private val sorts = new mutable.HashMap[String,Sort]()
  private val isARelations = collection.mutable.Map[Sort,Set[Sort]]().withDefaultValue(Set[Sort]())

  private val predicatesByName = new mutable.HashMap[String,AbstractPredicateDefinition]()
  private val constantsBySortMutable = new mutable.HashMap[Sort,Set[Constant]]()

  //todo get rid of the constants map; because it doesn't allow sorted overloading of constants
  val constants = new mutable.HashMap[String,Constant]

  /**
   * Add a sub-super sort relation. The sorts get created if they are not existing yet.
   */
  def addIsA(subSort: String, superSort: String) {
    val subS = getOrCreateSort(subSort)
    val superS = getOrCreateSort(superSort)

    val oldSet = isARelations(subS)
    isARelations.update(subS,oldSet + superS)
  }

  def definePredicate(pname: String, sortStrings: List[String], functional: Boolean = false): Either[String,AbstractPredicateDefinition] = {
    if(predicatesByName.contains(pname)) return Left("defining a predicate that is already defined (%s)" format pname)

    val pred =  if(!functional){
      new NormalPredicateDefinition(
        pname,
        sortStrings.map(ss => sorts.getOrElseUpdate(ss,new Sort(ss)))
      )
    } else {
      new FunctionalPredicateDefinition(
        pname,
        sortStrings.map(ss => sorts.getOrElseUpdate(ss,new Sort(ss)))
      )
    }

    predicatesByName.put(pname, pred)
    Right(pred)
  }

  def predicateByName(pname: String): Option[AbstractPredicateDefinition] = predicatesByName.get(pname)
  def predicates: Iterable[AbstractPredicateDefinition] = predicatesByName.values

  def getOrCreateSort(sort_string: String): Sort = sorts.getOrElseUpdate(sort_string, Sort(sort_string))

  def defineConstants(sort: Sort, consts: List[Constant]) {
    require(consts.forall(_.sort == sort), "cannot add a constant of different sort as member of sort %s".format(sort.name))
    for(c <- consts) constants.put(c.name, c)
    constantsBySortMutable.put(sort,consts.toSet)
  }
  def defineConstants(sort_string: String, const_strings: List[String]) {
    val sort = getOrCreateSort(sort_string)
    val consts = const_strings.map(Constant(_,sort))
    defineConstants(sort, consts)
  }

  def defineConstantRange(sort_string: String, lower: Int, upper: Int) {
    val sort = getOrCreateSort(sort_string)
    val consts = (lower to upper).map(num => num -> Constant(num.toString, sort))
    consts.foreach(t => constants.put(t._1.toString,t._2))
    constantsBySortMutable.put(sort,consts.map(_._2).toSet)
  }
  def createPredicateInstance(pname: String, args: List[Term]) = Predicate(predicatesByName(pname), args)

  def getSignature: Signature = Signature(
    sorts.values.toSet,
    predicatesByName.values.toSet,
    constantsBySortMutable.toMap,
    isARelations.toMap
  )

  //implement AtomBuilder stuff
  def constantsBySort: Map[Sort, Set[Constant]] =
    Signature.buildTransitiveConstantMap(isARelations.toMap,constantsBySortMutable.toMap,sorts.values.toSet)

  def abstractPredDefs: Set[AbstractPredicateDefinition] = predicatesByName.values.toSet
}