package edu.uulm.scbayes.inference.sampling.sat

import org.specs2.mutable._
import scala.util.Random
import edu.uulm.scbayes.mln._
import parsing.ParseHelpers._
import edu.uulm.scbayes.logic._
import edu.uulm.scbayes.logic.cnf._
import parsing.SignatureBuilder
import sampling.{SampleSAT, RejectionSAT}
import edu.uulm.scbayes.util._

/**
 * Unit tests for the SampleSAT implementations.
 *
 *
 * Date: 25.03.11
 */

class SampleSATTest extends Specification {


  "1000 samples from RejectionSAT" should {

    implicit val sm = new SignatureBuilder

    parsePredicateDefs("p(A)")
    parseDomains("A={A,B,C,D,E,F}")

    val formulas = parseFormulas(
      "p(A) v !p(B)" +
        "p(B) v p(D) v !p(A)" +
        "!p(C) v !p(D)" +
        "!p(E) v p(F) v !p(A)" +
        "p(F) v p(D)", sm.getSignature)

    val clauses = formulas.map {
      case dj: Disjunction => dj
      case f => Disjunction(Set(f))
    }.map(GroundCNF.disjunction2Clause)

    val atoms: List[PredicateAtom#BaseType] = (for(cl <- clauses; a <- cl.atoms) yield a).collect{case p: PredicateAtom => p.base}

    val interpretations: Array[TruthAssignment] =
      crossProduct(atoms.toList.map(a => List(false, true))).iterator
        //each "interpretation" is now a list of boolean lining up with 'atoms'; convert to set of atoms
        .map(tf_list => atoms.zip(tf_list).toMap)
        .toArray
        .map(
          new TruthAssignment(_,Map.empty)
        )

    val satisfying_interpretations = interpretations.filter(interp => clauses.forall(_.evaluate(interp)))

    val sampler = new RejectionSAT

    val results: Seq[TruthAssignment] = (for (run <- 1 to 1000) yield
      sampler.sampleComplete(
        GroundCNF.fromClauses(clauses),
        sm.getSignature,
        sm.getSignature.herbrandBase.toSet,
        new Random(42))
      )

      //discard the empty Options
      .flatten

    val distinct = results.distinct

    "contain all solutions" in {
      satisfying_interpretations.forall {
        i1 => distinct.exists(sm.getSignature.interpretationsEqual(i1, _))
      } must beTrue
    }

    "contain only solutions" in {
      distinct.forall {
        i1 => satisfying_interpretations.exists(sm.getSignature.interpretationsEqual(i1, _))
      } must beTrue
    }
  }

  "query with no clauses and a signature with a functional predicate" should {

    implicit val sm = new SignatureBuilder

    parsePredicateDefs("p(A) f(A!)")
    parseDomains("A={A,B}")

    "yield a TruthAssignment for all groundpredicates" in {
      val random = new Random(42)
      val sampler = new RejectionSAT
      val ta = sampler.sampleComplete(
        GroundCNF.fromClauses(Seq()),
        sm.getSignature,
        sm.getSignature.herbrandBase.toSet,
        random).get
      sm.getSignature.normalBase.forall(base => ta.truth.isDefinedAt(base)) must beTrue
    }
  }

  "cnf must be split correctly" in {

    implicit val sm = new SignatureBuilder

    parsePredicateDefs("p(A)")
    parseDomains("A={A,B,C,D,E,F,G}")

    val formula = parseFormula("(p(A) v p(B)) ^ !p(C)", sm.getSignature)

    val cnf = GroundCNF.fromConjunction(formula.asInstanceOf[Conjunction])

    val sampleSat = new SampleSAT

    val split = sampleSat.splitCNF(cnf)

    split.size must beEqualTo(2)
  }
}
