package edu.uulm.scbayes.logic

import cnf.{ConjunctiveNormalForm, GroundCNF}
import org.specs2.mutable._

import edu.uulm.scbayes.mln.parsing.{ParseHelpers, SignatureBuilder}
import ParseHelpers._

/**
 * Specification file for operations on first-order formulas.
 *
 *
 * Date: 15.03.11
 * Time: 15:25
 */

class FOLOpsTest extends Specification {
  import FormulaDomain1._

  args(sequential = true)

  "substitution in predicate" in {
    val sortX: Sort = new Sort("X")
    val sortY: Sort = new Sort("Y")
    val pd = new NormalPredicateDefinition("p", List(sortX, sortY))
    val vX: Variable = new Variable("?x")
    val vY: Variable = new Variable("?y")
    val p = Predicate(pd, List(vX, vY))
    val const: Constant = new Constant("t", sortX)
    val subst = p.substitute(vX, const)

    subst must be_==(Predicate(pd, List(const, vY)))
  }

  "flatten conjunction" in {
    val f1 = "p(?z) ^ (p(?x) ^ p(?y))".toFormula
    val f2 = "p(?z) ^ p(?x) ^ p(?y)".toFormula

    FormulaOps.flatten(f1.asInstanceOf[Conjunction]) must beEqualTo(f2)
  }

  "flatten disjunction" in {
    val f1 = "p(?z) v (p(?x) v p(?y))".toFormula
    val f2 = "p(?z) v p(?x) v p(?y)".toFormula

    FormulaOps.flatten(f1.asInstanceOf[Disjunction]) must beEqualTo(f2)
  }

  "simplifying nested conjunctions" in {
    val simplified = "p(?z) v (p(?x) v p(?y))".toFormula.simplify
    simplified must_== "p(?z) v p(?x) v p(?y)".toFormula
  }

  "simplifying deeply nested disjunctions" in {
    val simplified = "p(?z) ^ (p(?x) ^ (p(?y) ^ p(?w)))".toFormula.simplify
    simplified must_== "p(?z) ^ p(?x) ^ p(?y) ^ p(?w)".toFormula
  }

  "simplifying double negation" in {
    "!(!(!p(?x)))".toFormula.simplify must_== "!p(?x)".toFormula
  }

  "simplifying deeply nested disjunctions with negations" in {
    "p(?w) v (p(?x) v (!p(?y) v p(?z)))".toFormula.simplify must_== "p(?w) v p(?x) v !p(?y) v p(?z)".toFormula
  }

  "converting simple formula to CNF" in {
    GroundCNF.formulaToCNF("p(?x) v (p(?y) ^ p(?z))".toFormula) must_== "(p(?x) v p(?y)) ^ (p(?x) v p(?z))".toFormula
  }

  "'p(?x) v !((p(?y) ^ p(?z)))' must become 'p(?x) v (!p(?y) v !p(?z))' when negations are pushed" in {
    "p(?x) v !(p(?y) ^ p(?z))".toFormula.pushNegations === "p(?x) v (!p(?y) v !p(?z))".toFormula
  }
  "converting simple formula with negation to CNF" in {
    //we need to manually add the conjunction around the second formula
    GroundCNF.formulaToCNF("p(?x) v !(p(?y) ^ p(?z))".toFormula) must beEqualTo(Conjunction(Set("p(?x) v !p(?y) v !p(?z)".toFormula)))
  }

  "distributive law for (a v (b ^ c))" in {
    val f1 = "p(?x) v (p(?y) ^ p(?z))".toFormula
    val f2 = "(p(?x) v p(?y)) ^ (p(?x) v p(?z))".toFormula

    FormulaOps.distribute(f1.asInstanceOf[Disjunction]) must beEqualTo(f2)
  }

  "distributive law for (a v (b ^ c) v (d ^ e))" in {
    val f1 = "p(?x) v (p(?y) ^ p(?z)) v (p(?a) ^ p(?b))".toFormula
    val f2 = "(p(?x) v p(?y) v p(?a)) ^ (p(?x) v p(?z) v p(?a)) ^ (p(?x) v p(?y) v p(?b)) ^ (p(?x) v p(?z) v p(?b))".toFormula

    val f1Disctributed = FormulaOps.distribute(f1.asInstanceOf[Disjunction])
    f1Disctributed must_== f2
  }

  "formula traversal must find a predicate" in {
      "p(?x) v (p(?y) ^ p(?z))".toFormula.extract must contain("p(?y)".toFormula)
  }

  "finding out the sort of a variable" in {
    "p(?x)".toFormula.getTypeOf(Variable("?x")) must beEqualTo(Sort("P"))
  }

  "finding all free variables in a formula" in {
    implicit val sm = new SignatureBuilder
    val f = "p(?x) ^ !(p(?y) => (EXISTS ?z p(?z)))".toFormula
    f.freeVariables must beEqualTo(Set(Variable("?x"), Variable("?y")))
  }

  "finding a predicate below a quantifier" in {
    val f = "FORALL ?y (p(?y) v p(?x))".toFormula
    f.extract.collect{case p: Predicate => p} must not be empty
    f.extract.size must be_==(4)
  }

  "unpacking a disjunction" in {
    val f = "(p(?x) v p(?y))".toFormula
    f.extract.collect{case p: Predicate => p}.size must be_==(2)
    f.extract.size must be_==(3)
  }

  "pushing negation for a doubly negated literal should eliminate them" in {
    "!(!p(?x))".toFormula.pushNegations === "p(?x)".toFormula
  }

  "pushing a negation past a quantifier" in {
    "!(EXISTS ?v p(?v))".toFormula.pushNegations === "FORALL ?v !p(?v)".toFormula
  }

  "formula to cnf" in {
    GroundCNF.formulaToCNF("(p(a) ^ p(b)) <=> p(c)".toFormula) must beEqualTo("(!p(a) v p(c)) ^ (!p(b) v p(c)) ^ (p(a) v p(b) v !p(c))".toFormula)
  }

  "evaluating p(A1) ^ !p(A2) for different interpretations" in {
    implicit val sm = new SignatureBuilder
    parsePredicateDefs("p(A)")
    parseDomains("A={A1,A2}")
    val f1 :: f2 :: _ = parseFormulas(
      "p(A1) ^ !p(A2)" +
        "p(A1) v p(A2)", sm.getSignature
    )

    val interpretation1 = { gp: GroundPredicate =>
      gp.parameters.head.toString match {
        case "A1" => true
        case "A2" => false
      }
    }

    val interpretation2 = { gp: GroundPredicate =>
      gp.parameters.head.toString match {
        case "A1" => false
        case "A2" => false
      }
    }
    f1.evaluate(interpretation1) must beTrue
    f1.evaluate(interpretation2) must beFalse

    f2.evaluate(interpretation1) must beTrue
    f2.evaluate(interpretation2) must beFalse
  }

  "grounding a formula's free variables" in {
    implicit val sm = new SignatureBuilder
    parsePredicateDefs("p(A)")
    parseDomains("A={A1,A2,A3}")
    val f = parseFormula("FORALL ?x p(?y) ^ !p(?x)", sm.getSignature)

    val groundings = f.groundifyFreeVariables(sm.getSignature.constants)
    groundings.size must beEqualTo(3)
    groundings must contain(parseFormula("FORALL ?x p(A3) ^ !p(?x)", sm.getSignature))
  }

  "grounding a EXISTS quantifier" in {
    implicit val sm = new SignatureBuilder
    parsePredicateDefs("p(A)")
    parseDomains("A={A1,A2,A3}")
    val f = parseFormula("EXISTS x !p(x)", sm.getSignature)

    val grounding = f.groundifyQuantifiers(sm.getSignature.constants)
    grounding === parseFormula("!p(A1) v !p(A2) v !p(A3)", sm.getSignature)
  }

  "grounding a FORALL quantifier" in {
    implicit val sm = new SignatureBuilder
    parsePredicateDefs("p(A)")
    parseDomains("A={A1,A2,A3}")
    val f = parseFormula("FORALL x (!p(x) v p(A1))", sm.getSignature)

    val grounding = f.groundifyQuantifiers(sm.getSignature.constants)
    grounding === parseFormula("(!p(A1) v p(A1)) ^ (!p(A2) v p(A1)) ^ (!p(A3) v p(A1))", sm.getSignature)
  }


  "grounding a more complicated formula's free variables" in {
    implicit val sm = new SignatureBuilder
    parsePredicateDefs("p(A) q(B)")
    parseDomains("A={A1,A2,A3} B={B1,B2}")
    val f = parseFormula("FORALL ?x (p(?y) ^ !p(?x)) => (q(?z) v p(?w))", sm.getSignature)

    val groundings = f.groundifyFreeVariables(sm.getSignature.constants)
    //2 x A (?y,?w) and 1 x B (?z) = 3 * 3 * 2 = 18
    groundings.size must beEqualTo(18)
    groundings must contain(parseFormula("FORALL ?x (p(A1) ^ !p(?x)) => (q(B2) v p(A3))", sm.getSignature))
  }

  "tautology testing for a v !a" in {
    FormulaOps.isTautology("p(?x) v !p(?x)".toFormula.asInstanceOf[Disjunction]) must beTrue
  }

  "tautology testing for a v !b" in {
    FormulaOps.isTautology("p(?x) v !p(?y)".toFormula.asInstanceOf[Disjunction]) must beFalse
  }

  "tautology testing for a v a" in {
    FormulaOps.isTautology("p(?x) v p(?x)".toFormula.asInstanceOf[Disjunction]) must beFalse
  }
}