package edu.uulm.scbayes.mln

import org.specs2.mutable._

import edu.uulm.scbayes.experiments.examples.ExampleGraphs
import edu.uulm.scbayes.factorgraph._
import edu.uulm.scbayes.mln.factorgraph._

/**
 * Contains structural tests for whole MLNs in form of factor graphs.
 *
 *
 * Date: 21.03.11
 */

class MLNFactorGraphTest extends Specification {
  "a simple MLN (Smokers)" should {

    val graph = ExampleGraphs.smokers

    "contain 6 variables" in {
      graph.variables.size must beEqualTo(6)
    }

    "must contain 6 factors" in {
      //used to be 6, but became 4 since pruning constant factors
      graph.factors.size must beEqualTo(4)
    }

    "cancer(PETER) must have one factor" in {
      val smokes_peter = graph.variables.find(v => v.toString == "cancer(PETER)").get
      graph.factorsOf(smokes_peter).size must beEqualTo(1)

    }
    "must contain a certain grounding: 'smokes(PETER) => cancer(PETER)'" in {
      graph.factors.map(_.formula.toString) must contain("cancer(PETER) v !smokes(PETER)")
    }

    def mustShareAFactor[A,B](graph: FactorGraph[A,B], varNames: String*) = {
      val variables = varNames.map(vn => graph.variables.find(_.toString == vn).get)
      val factorSets = variables.map(graph.factorsOf)

      //intersect all
      factorSets.reduce(_ intersect _) must not be empty
    }

    "smokes(PETER) and friends(PETER,THOMAS) must share a factor" in {
      mustShareAFactor(graph, "smokes(PETER)", "friends(PETER,THOMAS)")
    }

    "cancer(PETER) and smokes(PETER) must share a factor" in {
      mustShareAFactor(graph, "cancer(PETER)", "smokes(PETER)")
    }
  }

  "a simple functional MLN" should {
    val graph: MLNFactorGraph = ExampleGraphs.simpleFunctional
    "contain 8 variables in total" in {
      graph.variables.size must beEqualTo(8)
    }
    "contain 2 functional predicates" in {
      graph.variables.count(_.isInstanceOf[FunctionValuedNode]) must beEqualTo(2)
    }
    "contain 6 normal predicates" in {
      graph.variables.count(_.isInstanceOf[PredicateNode]) must beEqualTo(6)
    }
  }

  "factor graph constructed with evidence may not contain any evidence variable" in {
    val graph = ExampleGraphs.smokersWithEvidence
    val evidence = ExampleGraphs.smokersEvidence

    val evidenceNodes = (evidence.truth.keySet ++ evidence.functionValue.keySet).map(LogicNode(_, graph.signature))
    val representedEvidenceNodes = graph.variables intersect evidenceNodes

    representedEvidenceNodes must beEmpty
  }
}