package edu.uulm.scbayes.mln.parsing

import org.specs2.mutable._

import FOLParser._
import ParseHelpers._
import edu.uulm.scbayes.logic._
import edu.uulm.scbayes.experiments.examples.ExampleGraphs

/**
 * Tests for the first-order formula parser.
 *
 * Date: 3/11/11
 */

class FOLParseTest extends Specification{


  "with simple signature a(X,Y)" should {
    implicit val sm = new SignatureBuilder
    //this line serves to add the 'a' predicate to sm
    parsePredicateDefs("a(X,Y)")
    val signature = sm.getSignature

    "a must have two parameters"  in {
      sm.predicateByName("a").get.parameters must have size(2)
    }

    "implication must be resolved" in {
      val f = parseFormulas(
        "a(?x,?y) => a(?x, ?z) " +
        "!a(?x,?y) v a(?x,?z)",
        signature
      )

      f(0) must beEqualTo(f(1))
    }

    "equivalence must be resolved" in {
      val f = parseFormulas(
        "a(?x,?y) <=> a(?x, ?z) " +
        "(a(?x,?y) ^ a(?x,?z)) v (!a(?x,?y) ^ !a(?x,?z))",
        signature
      )
      val equiv = f(0)
      val expanded = f(1)

      equiv must_== expanded
    }

    "parsing formula with weight" in {
      val parseResult = FOLParser.parse(weightedFormula(sm), "0.5 a(?e,?f)")
      parseResult.get.weight must beSome(0.5d)
    }

    "parsing formula with variable constraints" in {
      val parseResult = FOLParser.parse(weightedFormula(sm), "0.5 a(e,f) [e != f]")
      parseResult.get.variableInequalities === Set((Variable("e"),Variable("f")))
    }

    "parsing of rigid formula" in {
      val parseResult = FOLParser.parse(weightedFormula(sm), "a(?e,?f).")
      parseResult.get.weight must beNone
    }

    "parsing a sort domain" in {
      val parseResult = FOLParser.parse(domainDefinition,"A = {Rolf, Peter, Kurt}")
      parseResult.successful must beTrue
      val sort: Sort = Sort("A")
      sm.constantsBySort(sort) must contain(Constant("Peter",sort))
    }

  }

  "parsing the simple temporal domain should result in one functional predicate definition" in {
    val (signature,_) = parseMLN(ExampleGraphs.simpleTemporalSource)
    signature.predicates.count(_.isInstanceOf[FunctionalPredicateDefinition]) aka "number of functional predicates" must beEqualTo(1)
  }

  "parsing the simple temporal domain should result in one normal predicate definition" in {
    val (signature,_) = parseMLN(ExampleGraphs.simpleTemporalSource)
    signature.predicates.count(_.isInstanceOf[NormalPredicateDefinition]) aka "number of normal predicates" must beEqualTo(1)
  }
}
