package edu.uulm.scbayes.mln.parsing

import org.specs2._

import FOLParser._
import java.io.FileReader
import edu.uulm.scbayes.logic._

/**
 * See if we can parse a mln file which contains sort hierarchy statements.
 */

class SortHierarchyTest extends Specification{
  args(sequential = true)

  val testFile1 = "res/mln/sort-hierarchy/domain1.mln"

  def is =
    "Sort Hierarchy Tests".title ^
    "isA statements like 'A isA B' states that every object of sort A is also" ^
      " a member of sort A." ^
    p^
    "parsing single strings with isA relations" ^
      "simple 'A isA B' statement should be parseable"            ! simpleParseTest ^
      "creating a signature after parsing 'A isA B' should work"  ! createValidSignature ^
      "fail to build a Signature from two cyclic sort relations"  ! cyclesTest1 ^
    p^
    "parsing whole MLN file (domain1) with isA relations" ^
      "file must be parseable"                          ! parseSimpleFile ^
      "most primitive sort D has one constant"          ! testConstants("D",1) ^
      "sort B must have 1+1 constants (one from D)"     ! testConstants("B",1+1) ^
      "sort C must have 2+1 constants (one form D)"     ! testConstants("C",2+1) ^
      "sort A must have 4 constants"                    ! testConstants("A",4) ^
      "number of groundings for first formula must be 1"  ! groundFormula1


  def parseSimpleFile = {
    implicit val signatureBuilder = new SignatureBuilder
    val result = FOLParser.parseAll(mlnFileParser, new FileReader(testFile1))
    result.successful aka result.toString must beTrue
  }

  def groundFormula1 = {
    implicit val sm = new SignatureBuilder
    val (signature, formulaList) = FOLParser.parseAll(mlnFileParser, new FileReader(testFile1)).get

    formulaList.head.formula.groundifyFreeVariables(signature.constants) must have size(1)
  }

  /** Parse the simple file and check the number of constants of a given sort to be as specified. */
  def testConstants(sortName: String, numConstants: Int) = {
    implicit val signatureBuilder = new SignatureBuilder

    FOLParser.parseAll(mlnFileParser, new FileReader(testFile1))

    val signature = signatureBuilder.getSignature

    val sort = signature.sorts.find(_.name == sortName).get
    signature.constants(sort) aka "constants of type %s".format(sortName) must have size(numConstants)
  }

  def simpleParseTest = {
    val statement = "A isA B"
    implicit val signatureBuilder = new SignatureBuilder
    val result = FOLParser.parse(isAStatement, statement)
    result.successful aka result.toString must beTrue
  }

  def createValidSignature = {
    val statement = "A isA B"
    implicit val signatureBuilder = new SignatureBuilder
    FOLParser.parse(isAStatement, statement)

    signatureBuilder.getSignature must not throwA(new Exception)
  }

  def cyclesTest1 = {
    val statement1 = "A isA B"
    val statement2 = "B isA A"

    implicit val signatureBuilder = new SignatureBuilder

    FOLParser.parse(isAStatement, statement1)
    FOLParser.parse(isAStatement, statement2)

    signatureBuilder.getSignature must throwA[IllegalArgumentException]
  }
}