{-# LANGUAGE GADTs,
             RankNTypes,
             TypeSynonymInstances,
             FlexibleInstances,
             FlexibleContexts,
             MultiParamTypeClasses,
             KindSignatures,
             DataKinds #-}
--------------------------------------------------------------------------------
-- | This module exports a 'Boxed' data type and many functions to construct
-- boxed versions of statically typed data types, i.e. where the type parameters
-- are hidden.
module Flux.Typed.Boxed(
  -- *
  ErrorMsg, TypeVar, TypeName,
  -- * Boxed
  Boxed(..), boxed, boxedType,
  -- * Var
  Var(), Scope(..), mkVar, mkCons,
  -- * Type
  Type(), mkTypeUnit, mkType, mkTypeFun, mkTypeApply, mkTypeTuple,
  mkTypeVar, mkTypeForAll,
  -- * Lit
  Lit(), mkLitSymbol, mkLitPrimChar, mkLitPrimString, mkLitPrimInt, mkLitInteger,
  mkLitPrimDouble, mkLitPrimFloat,
  -- * Expr
  Expr(), mkVarE, mkLitE, mkLamE, mkTLamE, mkAppE, mkLetRecE,
  mkCaseE, mkUnifyAppE, mkUnifyCaseE, etaExpand,
  -- * Pat
  Pat(), mkPatUnit, mkPatVar, mkPatLit, mkPatApp, mkPatTuple, mkUnifyPatApp,
  -- * Binds
  Binds(), mkNoBind, mkBinds,
  -- * Type Classes
  ApplyType(..), ChangeType(..), changeSubType
) where

import Flux.Typed.Type
import Flux.Typed.Var
import Flux.Typed.Literal
import Flux.Typed.Expr
import Flux.Typed.Pattern
import Flux.Typed.Unification

import Control.Applicative
import Control.Monad
import Data.List(intercalate, union, delete)
import Data.Type.Equality
import Data.Proxy
import Data.Maybe(fromMaybe)
import GHC.TypeLits

import System.IO.Unsafe
--------------------------------------------------------------------------------
type ErrorMsg = String
type TypeVar  = String
type TypeName = String
--------------------------------------------------------------------------------
data Boxed (f :: TypeKind -> *) where Boxed :: (GenType t) => f t -> Boxed f

boxed :: (forall a. GenType a => f a -> b) -> Boxed f -> b
boxed f (Boxed t) = f t
--------------------------------------------------------------------------------
mkLitSymbol :: String -> Boxed Lit
mkLitSymbol s = Boxed (LitSymbol s)

mkLitPrimChar :: Char -> Boxed Lit
mkLitPrimChar c = Boxed (LitPrimChar c)

mkLitPrimString :: String -> Boxed Lit
mkLitPrimString s = Boxed (LitPrimString s)

mkLitPrimInt :: Integer -> Boxed Lit
mkLitPrimInt i = Boxed (LitPrimInt i)

mkLitInteger :: Integer -> Boxed Lit
mkLitInteger i = Boxed (LitInteger i)

mkLitPrimDouble :: Double -> Boxed Lit
mkLitPrimDouble d = Boxed (LitPrimDouble d)

mkLitPrimFloat :: Float -> Boxed Lit
mkLitPrimFloat f = Boxed (LitPrimFloat f)
--------------------------------------------------------------------------------
mkVar :: Boxed Type -> VarName -> Scope -> Boxed Var
mkVar (Boxed t) n s = Boxed (Var t n s)

mkCons :: Boxed Type -> VarName -> Scope -> Arity -> Boxed Var
mkCons (Boxed t) n s a = Boxed (Cons t n s a)
--------------------------------------------------------------------------------
mkTypeUnit  :: Boxed Type
mkTypeUnit = Boxed TypeUnit

mkType :: TypeName -> Boxed Type
mkType t | SomeSymbol p <- someSymbolVal t = Boxed (Type p)

mkTypeFun   :: Boxed Type -> Boxed Type -> Boxed Type
mkTypeFun (Boxed a) (Boxed b) = Boxed (TypeFun a b)

mkTypeApply :: Boxed Type -> Boxed Type -> Boxed Type
mkTypeApply (Boxed a) (Boxed b) = Boxed (TypeApply a b)

mkTypeTuple :: Boxed Type -> Boxed Type -> Boxed Type
mkTypeTuple (Boxed a) (Boxed b) = Boxed (TypeTuple a b)

mkTypeVar :: TypeVar -> Boxed Type
mkTypeVar v | SomeSymbol p <- someSymbolVal v = Boxed (TypeVar p)

mkTypeForAll :: TypeVar -> Boxed Type -> Boxed Type
mkTypeForAll v (Boxed t)
  | SomeSymbol p <- someSymbolVal v = Boxed (TypeForAll p t)
--------------------------------------------------------------------------------
mkVarE :: Boxed Var -> Boxed Expr
mkVarE (Boxed v) = Boxed (VarE v)

mkLitE :: Boxed Lit -> Boxed Expr
mkLitE (Boxed l) = Boxed (LitE l)

mkLamE :: Boxed Var -> Boxed Expr -> Boxed Expr
mkLamE (Boxed v) (Boxed e) = Boxed (LamE v e)

mkTLamE :: TypeVar -> Boxed Expr -> Boxed Expr
mkTLamE v (Boxed e) | SomeSymbol p <- someSymbolVal v = Boxed (TLamE p e)

mkAppE :: Boxed Expr -> Boxed Expr -> Either ErrorMsg (Boxed Expr)
mkAppE (Boxed e1) (Boxed e2)
  | TypeFun a _ <- typeOf e1,
    Just Refl <- testEquality a (typeOf e2) = Right (Boxed (AppE e1 e2))
  | TypeFun a _ <- typeOf e1 = Left $ "mkAppE: types do not match: \n" ++
      show e1 ++ " :: " ++ show (typeOf e1) ++ "\n" ++
      show e2 ++ " :: " ++ show (typeOf e2) ++ "\n"

  | otherwise = Left $ "mkAppE: first type must be a function type: \n" ++
      show e1 ++ " :: " ++ show (typeOf e1) ++ "\n" ++
      show e2 ++ " :: " ++ show (typeOf e2) ++ "\n"

mkUnifyAppE :: Boxed Expr -> Boxed Expr -> Either ErrorMsg (Boxed Expr)
mkUnifyAppE (Boxed e1) (Boxed e2)
  | TypeFun a _ <- typeOf e1,
    [s] <- unify (Boxed a) (Boxed (typeOf e2)) = do
      e1' <- changeType (s @@) e1
      e2' <- changeType (s @@) e2
      mkAppE e1' e2'

  | TypeFun a _ <- typeOf e1 = Left $ "mkUnifyAppE: cannot unify types: \n" ++
      show e1 ++ " :: " ++ show (typeOf e1) ++ "\n" ++
      show e2 ++ " :: " ++ show (typeOf e2) ++ "\n"

  | otherwise = Left $ "mkUnifyAppE: first type must be a function type: \n" ++
      show e1 ++ " :: " ++ show (typeOf e1) ++ "\n" ++
      show e2 ++ " :: " ++ show (typeOf e2) ++ "\n"


mkLetRecE :: Boxed Binds -> Boxed Expr -> Boxed Expr
mkLetRecE (Boxed bs) (Boxed e) = Boxed (LetRecE bs e)

mkCaseE :: Boxed Expr -> [(Boxed Pat, Boxed Expr)] -> Either ErrorMsg (Boxed Expr)
mkCaseE _ [] = Left "mkCaseE: empty case list"
mkCaseE (Boxed e) cs = do
  (lhs,rhs@(Boxed r : _)) <- return (unzip cs)
  lhs' <- sameTypes (typeOf e) lhs ("mkCaseE: patterns do not have the same type: \n" ++
              show e ++ " :: " ++ show (typeOf e) ++ "\n" ++
              intercalate "\n" (map (\(Boxed p) -> show p ++ " :: " ++ show (typeOf p)) lhs))
  rhs' <- sameTypes (typeOf r) rhs "mkCaseE: right-hand-side-expressions do not have the same type"
  return (Boxed (CaseE e (zip lhs' rhs')))
  where
    sameTypes :: Typed f => Type a -> [Boxed f] -> ErrorMsg -> Either ErrorMsg [f a]
    sameTypes t [] _ = Right []
    sameTypes t ((Boxed e):es) errorMsg
      | Just Refl <- testEquality t (typeOf e) = (e:) <$> sameTypes t es errorMsg
      | otherwise = Left errorMsg

mkUnifyCaseE :: Boxed Expr -> [(Boxed Pat, Boxed Expr)] -> Either ErrorMsg (Boxed Expr)
mkUnifyCaseE _ [] = Left "mkUnifyCaseE: empty case list"
mkUnifyCaseE e cs = do
  (lhs,rhs) <- return (unzip cs)
  case unifylist (map (boxed (Boxed . typeOf)) $ lhs)
                 (map (boxed (Boxed . typeOf)) $ replicate (length lhs) e) of
    [s] -> do
      lhs' <- mapM (\(Boxed p) -> changeType (s@@) p) lhs
      mkCaseE e (zip lhs' rhs)
    _ ->  Left $ "mkUnifyCaseE: could not unify patterns of cases: \n" ++
            intercalate "\n" (map (\(Boxed p) -> show p ++ " :: " ++ show (typeOf p)) lhs)

-- | makes sure that the expression is a lambda expression
etaExpand :: Boxed Expr -> Boxed Expr
etaExpand b@(Boxed e) = case e of
  LamE _ _   -> b
  TLamE _ e  -> etaExpand (Boxed e)
  otherwise  -> Boxed (LamE unit e)
--------------------------------------------------------------------------------
mkPatUnit  :: Boxed Pat
mkPatUnit = Boxed PatUnit

mkPatVar :: Boxed Var -> Boxed Pat
mkPatVar (Boxed v) = Boxed (PatVar v)

mkPatLit :: Boxed Lit -> Boxed Pat
mkPatLit (Boxed l) = Boxed (PatLit l)

mkPatApp :: Boxed Pat -> Boxed Pat -> Either ErrorMsg (Boxed Pat)
mkPatApp (Boxed p1) (Boxed p2)
  | TypeFun a _ <- typeOf p1,
    Just Refl <- testEquality a (typeOf p2) = Right (Boxed (PatApp p1 p2))
  | otherwise = Left $ "mkPatApp: could not match types of patterns: \n" ++
      show p1 ++ " :: " ++ show (typeOf p1) ++ "\n" ++
      show p2 ++ " :: " ++ show (typeOf p2)

mkUnifyPatApp :: Boxed Pat -> Boxed Pat -> Either String (Boxed Pat)
mkUnifyPatApp (Boxed p1) (Boxed p2)
  | TypeFun a b <- typeOf p1,
    [s] <- unify (Boxed a) (Boxed (typeOf p2)) = do
      p1' <- changeType (s@@) p1
      p2' <- changeType (s@@) p2
      mkPatApp p1' p2'

  | TypeFun a _ <- typeOf p1 = Left $ "mkUnifyPatApp: could not unify types: \n" ++
        show p1 ++ " :: " ++ show (typeOf p1) ++ "\n" ++
        show p2 ++ " :: " ++ show (typeOf p2) ++ "\n"

  | otherwise = Left $ "mkUnifyPatApp: first type must be a function type: \n" ++
        show p1 ++ " :: " ++ show (typeOf p1) ++ "\n" ++
        show p2 ++ " :: " ++ show (typeOf p2) ++ "\n"

mkPatTuple :: Boxed Pat -> Boxed Pat -> Boxed Pat
mkPatTuple (Boxed p1) (Boxed p2) = Boxed (p1 :* p2)
--------------------------------------------------------------------------------
mkNoBind :: Boxed Binds
mkNoBind = Boxed NoBind

mkBinds :: Boxed Var -> Boxed Expr -> Boxed Binds -> Either ErrorMsg (Boxed Binds)
mkBinds (Boxed v) (Boxed e) (Boxed b)
  | Just Refl <- testEquality (typeOf v) (typeOf e) = Right (Boxed (Binds v e b))
  | otherwise = Left "mkBinds: could not match types"
--------------------------------------------------------------------------------
instance Show (Boxed Type) where show = boxed show
instance Show (Boxed Var)  where show = boxed show
instance Show (Boxed Expr) where show = boxed show
instance Show (Boxed Pat)  where show = boxed show

instance Eq (Boxed Type) where
  Boxed t1 == Boxed t2
    | Just Refl <- testEquality t1 t2 = t1 == t2
    | otherwise                       = False

instance Eq (Boxed Pat) where
  Boxed p1 == Boxed p2
    | Just Refl <- testEquality p1 p2 = p1 == p2
    | otherwise                       = False
--------------------------------------------------------------------------------
boxedType :: Typed f => Boxed f -> Boxed Type
boxedType (Boxed f) = Boxed (typeOf f)
--------------------------------------------------------------------------------
class ApplyType f where
  -- instantiate a quantified (polymorphic) expression with a given type
  (@!) :: (GenType t) => f (TyForAll v t) -> Boxed Type -> Either String (Boxed f)

instance ApplyType Type where
  TypeForAll v t @! t' = Right $ (symbolVal v |-> t') @@ (Boxed t)

instance ApplyType Var where
  v @! t = case varType v @! t of
    Left errorMsg    -> Left errorMsg
    Right (Boxed t') -> Right (Boxed (changeVarType v t'))

instance ApplyType Expr where

  VarE v @! t = mkVarE <$> v @! t

  TLamE v e @! t = instantiateTypeVar v t e

  e@(AppE e1 e2) @! t | TypeForAll v _ <- typeOf e = do
    e1' <- instantiateTypeVar v t e1
    e2' <- instantiateTypeVar v t e2
    mkAppE e1' e2'

  e @! t = Left $ "@!: unexpected expression\n" ++ show e

instantiateTypeVar :: (ChangeType f, KnownSymbol v, GenType a) => Proxy v ->
                      Boxed Type -> f a -> Either ErrorMsg (Boxed f)
instantiateTypeVar v t = changeType f
  where
    f b@(Boxed (TypeVar u))
      | Just Refl <- sameSymbol u v = t
    f (Boxed (TypeForAll u t'))
      | Just Refl <- sameSymbol u v = f (Boxed t')
      | otherwise                   = mkTypeForAll (symbolVal u) (f (Boxed t'))
    f (Boxed (TypeFun a b))         = mkTypeFun (f (Boxed a)) (f (Boxed b))
    f (Boxed (TypeApply a b))       = mkTypeApply (f (Boxed a)) (f (Boxed b))
    f (Boxed (TypeTuple a b))       = mkTypeTuple (f (Boxed a)) (f (Boxed b))
    f b                             = b
--------------------------------------------------------------------------------
class ChangeType f where
  changeType :: GenType a => (Boxed Type -> Boxed Type) -> f a -> Either ErrorMsg (Boxed f)
--------------------------------------------------------------------------------
instance ChangeType Type where
  changeType f t = Right (f (Boxed t))

instance ChangeType Lit where
  changeType _ l = Right (Boxed l)

instance ChangeType Var where
  changeType f v = do
    Boxed t <- changeType f (varType v)
    return (Boxed (changeVarType v t))

instance ChangeType Pat where
  changeType _ PatUnit = Right mkPatUnit
  changeType f (PatVar v) = mkPatVar <$> changeType f v
  changeType f (PatLit l) = mkPatLit <$> changeType f l
  changeType f (PatApp p q) = do
    p' <- changeType f p
    q' <- changeType f q
    mkPatApp p' q'
  changeType f (p :* q) = mkPatTuple <$> changeType f p <*> changeType f q

instance ChangeType Binds where
  changeType f NoBind = Right mkNoBind
  changeType f (Binds v e b) = do
   v' <- changeType f v
   e' <- changeType f e
   b' <- changeType f b
   mkBinds v' e' b'

instance ChangeType Expr where
  changeType f (VarE v)   = mkVarE <$> changeType f v
  changeType f (LitE l)   = mkLitE <$> changeType f l
  changeType f (LamE v e) = mkLamE <$> changeType f v <*> changeType f e

  changeType f (AppE e1 e2) = do
    e1' <- changeType f e1
    e2' <- changeType f e2
    mkAppE e1' e2'

  changeType f (CaseE e cs) = do
    e' <- changeType f e
    cs' <- forM cs $ \(p,r) -> do
      p' <- changeType f p
      r' <- changeType f r
      return (p',r')
    mkCaseE e' cs'

  changeType f (TLamE v e) = mkTLamE (symbolVal v) <$> changeType f e

  changeType f (LetRecE bs e) = mkLetRecE <$> changeType f bs <*> changeType f e
--------------------------------------------------------------------------------
instance Occurrence TypeVar (Boxed Type) where
  freeVars (Boxed t) = freeVars' t
    where
      freeVars' :: Type t -> [TypeVar]
      freeVars' (TypeUnit)       = []
      freeVars' (Type t)         = []
      freeVars' (TypeFun a b)    = freeVars' a `union` freeVars' b
      freeVars' (TypeApply a b)  = freeVars' a `union` freeVars' b
      freeVars' (TypeTuple a b)  = freeVars' a `union` freeVars' b
      freeVars' (TypeVar v)      = [symbolVal v]
      freeVars' (TypeForAll v t) = delete (symbolVal v) (freeVars' t)

instance Unifiable TypeVar (Boxed Type) where

  unify t1 t2 = case (t1,t2) of
    (Boxed t1', Boxed t2') -> case (t1',t2') of
      (TypeUnit, TypeUnit) -> [emptySubst]
      ((Type Proxy), (Type Proxy))
        | Just Refl <- testEquality t1' t2' -> [emptySubst]
      (TypeFun a1 b1, TypeFun a2 b2)     -> unifylist [Boxed a1, Boxed b1] [Boxed a2, Boxed b2]
      (TypeApply a1 b1, TypeApply a2 b2) -> unifylist [Boxed a1, Boxed b1] [Boxed a2, Boxed b2]
      (TypeTuple a1 b1, TypeTuple a2 b2) -> unifylist [Boxed a1, Boxed b1] [Boxed a2, Boxed b2]

      (TypeVar x, TypeVar y) | Just Refl <- sameSymbol x y -> [emptySubst]
      (TypeVar v, _) | not (symbolVal v `occursIn` t2) -> [symbolVal v |-> t2]
      (_, TypeVar v) | not (symbolVal v `occursIn` t1) -> [symbolVal v |-> t1]

      (TypeForAll v1 t1'', TypeForAll v2 t2'')
        | symbolVal v1 == symbolVal v2 -> unify (Boxed t1'') (Boxed t2'')
        | otherwise -> unifylist [Boxed (TypeVar v1), Boxed t1''] [Boxed (TypeVar v2), Boxed t2'']

      _ -> []

  s @@ box@(Boxed t) = case t of
    TypeVar v       -> fromMaybe box (lookupSubst (symbolVal v) s)
    TypeForAll v t' -> case lookupSubst (symbolVal v) s of
        Just (Boxed (TypeVar u)) -> mkTypeForAll (symbolVal u) (s @@ (Boxed t'))
        _                        -> mkTypeForAll (symbolVal v) (s @@ (Boxed t'))
    TypeFun a b     -> mkTypeFun (s @@ (Boxed a)) (s @@ (Boxed b))
    TypeApply a b   -> mkTypeApply (s @@ (Boxed a)) (s @@ (Boxed b))
    TypeTuple a b   -> mkTypeTuple (s @@ (Boxed a)) (s @@ (Boxed b))
    _               -> box
--------------------------------------------------------------------------------
changeSubType :: (ChangeType f, GenType a) => Boxed Type -> Boxed Type -> f a -> Either ErrorMsg (Boxed f)
changeSubType t1 t2 = changeType f
  where
    f box@(Boxed t)
      | box == t1 = t2
      | otherwise = case t of
          TypeUnit  -> box
          Type _    -> box
          TypeVar _ -> box
          TypeFun a b
            | Boxed a' <- f (Boxed a),
              Boxed b' <- f (Boxed b) -> Boxed (TypeFun a' b')
          TypeApply a b
            | Boxed a' <- f (Boxed a),
              Boxed b' <- f (Boxed b) -> Boxed (TypeApply a' b')
          TypeTuple a b
            | Boxed a' <- f (Boxed a),
              Boxed b' <- f (Boxed b) -> Boxed (TypeTuple a' b')
          TypeForAll v t'
            | Boxed t'' <- f (Boxed t') -> Boxed (TypeForAll v t'')
--------------------------------------------------------------------------------
instance Occurrence VarName (Boxed Pat) where
  freeVars (Boxed p) = freeVars' p
    where
      freeVars' :: Pat t -> [VarName]
      freeVars' (PatVar v)      = [varName v]
      freeVars' (PatApp p1 p2)  = freeVars' p1 `union` freeVars' p2
      freeVars' (p1 :* p2)      = freeVars' p1 `union` freeVars' p2
      freeVars' (PatForAll _ p) = freeVars' p
      freeVars' _               = []

instance Unifiable VarName (Boxed Pat) where
  unify t1@(Boxed p1) t2@(Boxed p2) = case (p1,p2) of
    (PatUnit,PatUnit) -> [emptySubst]
    (PatLit l1, PatLit l2) | Just Refl <- testEquality l1 l2, l1 == l2 -> [emptySubst]
    (PatApp p1 p2, PatApp p1' p2') -> unifylist [Boxed p1, Boxed p2] [Boxed p1', Boxed p2']
    (p1 :* p2, p1' :* p2') -> unifylist [Boxed p1, Boxed p2] [Boxed p1', Boxed p2']
    (PatForAll v1 p1, PatForAll v2 p2)
      | Just Refl <- sameSymbol v1 v2 -> unify (Boxed p1) (Boxed p2)

    (PatVar x, PatVar y) | Just Refl <- testEquality x y, x == y -> [emptySubst]
    (PatVar v, _) | isVar v, not (varName v `occursIn` t2) -> [varName v |-> t2]
    (_, PatVar v) | isVar v, not (varName v `occursIn` t1) -> [varName v |-> t1]
    _ -> []

  s @@ (Boxed (PatVar v)) | Just p <- lookupSubst (varName v) s = p
  s @@ (Boxed (PatApp p1 p2)) | Right p <- mkPatApp (s @@ Boxed p1) (s @@ Boxed p2) = p
  s @@ (Boxed (p1 :* p2)) = mkPatTuple (s @@ Boxed p1) (s @@ Boxed p2)
  s @@ (Boxed (PatForAll v p)) | Boxed p' <- s @@ (Boxed p) = Boxed (PatForAll v p')
  s @@ p = p
--------------------------------------------------------------------------------
