{-# LANGUAGE NamedFieldPuns #-}
{-# LANGUAGE GADTs #-}
----------------------------------------------------------------------------
module Flux.Typed.GraphViz(graphToDot) where

import Data.Maybe(fromMaybe)
import Data.List(intercalate, nubBy)
import Data.Function(on)

import Flux.Typed.Type
import Flux.Typed.Boxed
import Flux.Typed.Graph
import Flux.Util(escapeHtml)
--------------------------------------------------------------------------------
type Attribute = (String,String)
--------------------------------------------------------------------------------
attributeList :: [Attribute] -> String
attributeList [] = ""
attributeList xs = "[" ++ intercalate "," (map showAttribute xs) ++ "]"

showAttribute :: Attribute -> String
showAttribute (attr, val) = attr ++ "=" ++ "\"" ++ val ++ "\""

dotStatement :: String -> [Attribute] -> String
dotStatement statement attributes = statement ++ " " ++ attributeList attributes

statements :: [String] -> String
statements = intercalate ";\n"

uniteAttributes :: [Attribute] -> [Attribute]
uniteAttributes = nubBy ((==) `on` fst)
--------------------------------------------------------------------------------
-- | compute a string in the Graphviz dot format from our graph data structure
graphToDot :: Graph -> String
graphToDot g = concat
  [
  case graphType g of
    DGraph   -> "digraph "
    SubGraph -> "subgraph cluster", show (graphId g), " {\n",

  statements . concat $
    [
      [dotStatement "graph" graphAttributes],
      map graphToDotNode (nodes g),
      map graphToDotEdge (edges g),
      map graphToDot (subgraphs g)
    ],

  "}\n"
  ]
  where
    graphAttributes = typeSpecificAttributes ++
      [("rankdir","TB"),("compound","true"), ("splines","true"),
       ("bgcolor","transparent"),("ranksep","0.4"),("nodesep","0.5"),
       ("concentrate","true"),("mindist","5.0")]
      where
        typeSpecificAttributes = case graphType g of
          DGraph   -> []
          SubGraph -> [("style","dashed"),("fillcolor","white"),("color","lightslategray")]
--------------------------------------------------------------------------------
isSource, isSink :: NodeType -> Bool

isSource NodeStart    = True
isSource NodeStartInv = True
isSource _            = False

isSink NodeEnd    = True
isSink NodeEndInv = True
isSink _          = False
--------------------------------------------------------------------------------
graphToDotNode :: Node -> String
graphToDotNode (Node {nodeId, nodeType, nodeIn, nodeOut})
  | isSource nodeType = "{ rank = source " ++ nodeOutput ++ " }"
  | isSink nodeType   = "{ rank = sink "   ++ nodeOutput ++ " }"
  | otherwise         = nodeOutput
  where
    nodeTypeAttributes NodeId          = [("label","id"),("style", "filled"),("shape","box"),("fillcolor","white"),("height", "0.2"),("width","0.2")]
    nodeTypeAttributes NodeStart       = [("label",""),("style", "filled"),("shape","invhouse"),("color","white"),("fillcolor","black"),("height", "0.2"),("width","0.2")]
    nodeTypeAttributes NodeEnd         = [("label",""),("style", "filled"),("shape","invhouse"),("color","white"),("fillcolor","black"),("height", "0.2"),("width","0.2")]
    nodeTypeAttributes (NodePattern p) = [("label", escapeHtml (show p)),("shape","box"),("style","dashed,filled"),("fillcolor","white"),("color","dimgray")]
    nodeTypeAttributes (NodeArr v)     = [("label", escapeHtml (show v)),("shape","box"),("style","filled"),("fillcolor","lightsteelblue"),("color","white"),("fontcolor","midnightblue")]
    nodeTypeAttributes (NodeLiteral l) = [("label", escapeHtml (show l)),("shape","ellipse"),("style","filled"),("fillcolor","gold") ,("color","white"),("fontcolor", "darkorange4")]
    nodeTypeAttributes (NodeVar v)     = [("label", escapeHtml (show v)),("shape","ellipse"),("style","filled"),("fillcolor","gold") ,("color","white"),("fontcolor", "darkorange4")]
    nodeTypeAttributes NodeApp         = [("label", ""),("fontsize", "0"),("width","0.3"),("height","0.3"),("shape","box"),("style","filled"),("fillcolor","lightsteelblue"),("color","white")]
    nodeTypeAttributes (NodeCase ps)   = [("label", intercalate "|" [ "<case" ++ escapeHtml (show k) ++ "> " ++ escapeHtml (show p) | (p,k) <- zip ps [1 .. length ps]]),("shape","Mrecord"),("style","solid,filled"),("fillcolor","darkslategray"),("color","white"),("fontcolor","white")]
    nodeTypeAttributes (NodeJoin l)    = [("label", intercalate "|" [ "<join" ++ escapeHtml (show k) ++ "> " | k <- [1..l]]), ("style", "filled"),("shape","record"),("fillcolor","black"),("height","0.01"),("fontsize","0")]
    nodeTypeAttributes NodeStartInv    = [("label",""),("style", "invisible"),("shape","box"),("height", "0.1"),("width","0.1")]
    nodeTypeAttributes NodeEndInv      = [("label",""),("style", "invisible"),("shape","box"),("height", "0.1"),("width","0.1")]

    nodeAttributes = ("fontname","DejaVu Sans Mono") : nodeTypeAttributes nodeType
    nodeOutput = case (nodeIn, nodeOut) of
      (Nothing,Nothing) -> dotStatement (show nodeId) nodeAttributes
      _ ->
        let i            = fromMaybe 0 nodeIn
            o            = fromMaybe 0 nodeOut
            label        = fromMaybe "" (lookup "label" nodeAttributes)
            inputFields  = if i < 1 then "" else  "{" ++ intercalate "|" [ "<in"  ++ show k ++ ">" | k <- [1..i] ] ++ "}|"
            outputFields = if o < 1 then "" else "|{" ++ intercalate "|" [ "<out" ++ show k ++ ">" | k <- [1..o] ] ++ "}"
            newLabel     = "{" ++ inputFields ++ label ++ outputFields ++ "}"
        in dotStatement (show nodeId) (uniteAttributes (("shape","record"):("label",newLabel):nodeAttributes))
--------------------------------------------------------------------------------
graphToDotEdge :: Edge -> String
graphToDotEdge e | Ports (srcPort, dstPort) <- edgePorts e =
    let edgeAttributes =
          let srcPat = portData srcPort
              dstPat = portData dstPort
              label  = show (typeOf srcPat)
              arrowhead = case portField dstPort of
                Just "w" -> "diamond"
                _        -> "open"
              srcCluster = case portCluster srcPort of
                Nothing -> []
                Just c  -> [("ltail","cluster" ++ show c)]
              dstCluster = case portCluster dstPort of
                Nothing -> []
                Just c  -> [("lhead","cluster" ++ show c)]
          in case typeOf srcPat of
               TypeCtrl -> uniteAttributes . concat $ [srcCluster, dstCluster,
                    [("dir","both"),("arrowsize","0.8"),("arrowhead","none"),
                    ("arrowtail","none"),("color","dimgray"),("style","dotted")]]
               _        -> uniteAttributes . concat $ [srcCluster, dstCluster,
                    [("label", escapeHtml label),("dir","both"),("arrowsize","0.8"),
                    ("arrowhead",arrowhead),("arrowtail","none"),("color","dimgray"),
                    ("fontsize","12"),("fontname","DejaVu Sans Mono"),("fontcolor","dimgray"),
                    ("labelfloat","false")]]
        src      = show (portId srcPort)
        dst      = show (portId dstPort)
        srcField = maybe "" (':':) (portField srcPort)
        dstField = maybe "" (':':) (portField dstPort)
    in dotStatement (src ++ srcField ++ ":s->" ++ dst ++ dstField ++ ":n") edgeAttributes
--------------------------------------------------------------------------------
