{-# LANGUAGE GADTs,
             Rank2Types,
             ScopedTypeVariables,
             TypeOperators,
             KindSignatures,
             DataKinds,
             StandaloneDeriving #-}
--------------------------------------------------------------------------------
module Flux.Typed.Pattern(Pat(..), traverseVars) where
--------------------------------------------------------------------------------
import Flux.Typed.Var
import Flux.Typed.Literal
import Flux.Typed.Type

import GHC.TypeLits
import Control.Applicative
import Data.Proxy
import Data.Type.Equality
import Data.List(intersperse)
--------------------------------------------------------------------------------
-- | Patterns can appear in case-of-expressions, and are used to represent
-- data-flows. Available constructs are:
-- unit types, variables, literals, pattern-application (used for ADTs), and tuples (:*).
data Pat :: TypeKind -> * where
  PatUnit   :: Pat TypeUnit
  PatVar    :: (GenType t)                => Var t -> Pat t
  PatLit    :: (GenType t)                => Lit t -> Pat t
  PatApp    :: (GenType a, GenType b)     => Pat (a :-> b) -> Pat a -> Pat b
  (:*)      :: (GenType a, GenType b)     => Pat a -> Pat b -> Pat (TyTuple a b)
  PatForAll :: (KnownSymbol v, GenType t) => Proxy (v :: Symbol) -> Pat t -> Pat (TyForAll v t)
----------------------------------------------------------------------------------
traverseVars :: Applicative f => (forall a. GenType a => Var a -> f (Var a)) -> Pat t -> f (Pat t)
traverseVars f p =
  case p of
    PatVar v     -> PatVar <$> f v
    PatApp p1 p2 -> PatApp <$> traverseVars f p1 <*> traverseVars f p2
    p1 :* p2     -> (:*)   <$> traverseVars f p1 <*> traverseVars f p2
    _            -> pure p
----------------------------------------------------------------------------------
instance Eq (Pat t) where
  PatUnit       == PatUnit             = True
  PatVar v1     == PatVar v2           = v1 == v2
  PatLit l1     == PatLit l2           = l1 == l2
  PatApp p1 p2  == PatApp p1' p2'
    | Just Refl <- testEquality p2 p2' = p1 == p1' && p2 == p2'
  p1 :* p2      == p1' :* p2'          = p1 == p1' && p2 == p2'
  PatForAll v1 p1 == PatForAll v2 p2   = v1 == v2 && p1 == p2
  _ == _                               = False
----------------------------------------------------------------------------------
instance TestEquality Pat where testEquality p1 p2 = testEquality (typeOf p1) (typeOf p2)
----------------------------------------------------------------------------------
instance Typed Pat where
  typeOf PatUnit = TypeUnit
  typeOf (PatVar v) = typeOf v
  typeOf (PatLit l) = typeOf l
  typeOf (PatApp p _) = case typeOf p of TypeFun _ b -> b
  typeOf (p :* q) = TypeTuple (typeOf p) (typeOf q)
  typeOf (PatForAll v p) = TypeForAll v (typeOf p)
---------------------------------------------------------------------------------
instance Show (Pat t) where
  showsPrec _ PatUnit    = showString "()"
  showsPrec _ (PatVar v) = shows v
  showsPrec _ (PatLit l) = shows l
  showsPrec _ (l :* r)   = showParen True (shows l . showChar ',' . shows r)
  showsPrec i x@(PatApp p q)
    | Just xs <- tuple shows x = showsTuple xs
    | Just xs <- list shows x  = showsList xs
    | otherwise = showParen (i > 0) $ shows p . showChar ' ' . showsPrec 1 q
    where
      showsTuple = showParen True . compose . intersperse (showChar ',')
      showsList xs = showChar '[' . compose (intersperse (showChar ',') xs) . showChar ']'
      compose = foldr (.) id
  showsPrec i (PatForAll v p) = showParen (i > 0) $ showString "/\\" . showString (symbolVal v) . showVars p
    where
      showVars :: forall a. Pat a -> ShowS
      showVars (PatForAll v' p) = showChar ' ' . showString (symbolVal v') . showVars p
      showVars p                = showChar '.' . shows p

tuple :: (forall a. Pat a -> b) -> Pat t -> Maybe [b]
tuple _ (PatVar v) | isTupleCons v = Just []
tuple f (PatApp p q) = do
    x <- tuple f p
    return (x ++ [f q])
tuple _ _ = Nothing

list :: (forall a. Pat a -> b) -> Pat t -> Maybe [b]
list _ (PatVar v) | isEmptyListCons v = Just []
list f (PatApp (PatApp (PatVar v) x) xs) | isListCons v = do
    xs' <- list f xs
    return (f x : xs')
list _ _ = Nothing
----------------------------------------------------------------------------------
isTupleCons :: Var t -> Bool
isTupleCons (Cons {varName = n, varScope = Global})
  | '(':xs <- n, [')'] <- dropWhile (==',') xs = True
isTupleCons _ = False

isListCons :: Var t -> Bool
isListCons (Cons {varName = n, varScope = Global}) = n == ":"
isListCons _ = False

--listCons :: Var (a -> [a] -> [a])
--listCons = Var (TypeFun (AnyType "a") (TypeFun ... TypeApp AnyType (AnyType "a"))) ":" Global

--emptyList :: Var [a]
--emptyList = Var AnyType "[]" Global

isEmptyListCons :: Var t -> Bool
isEmptyListCons (Cons {varName = n, varScope = Global}) = n == "[]"
isEmptyListCons _ = False
----------------------------------------------------------------------------------
instance VarOccurrence Pat where
  varOccursIn v (PatVar v') | Just Refl <- testEquality v v', v == v' = True
  varOccursIn v (PatApp p q) = varOccursIn v p || varOccursIn v q
  varOccursIn v (l :* r) = varOccursIn v l || varOccursIn v r
  varOccursIn v (PatForAll _ p) = varOccursIn v p
  varOccursIn _ _ = False
----------------------------------------------------------------------------------
