{-# LANGUAGE GADTs,
             StandaloneDeriving,
             KindSignatures,
             DataKinds,
             TypeOperators #-}
--------------------------------------------------------------------------------
module Flux.Typed.Type(Type(..), TypeKind(..),
  -- * Type Classes
  Typed(..), GenType(..),
  -- * Type Synonyms
  PrimInt, PrimChar, PrimString, PrimDouble, PrimFloat,
  TypeUnit, TypeInt, TypeChar, TypeInteger, TypeString, TypeFloat, TypeDouble,
  typePrimString, typePrimChar, typePrimInt, typePrimDouble, typePrimFloat,
  typeUnit, typeInt, typeChar, typeInteger, typeString, typeFloat, typeDouble) where
--------------------------------------------------------------------------------
import GHC.TypeLits
import Data.Proxy
import Data.Type.Equality
--------------------------------------------------------------------------------
-- | We don't need this data type directly, only the kind and the types promoted
-- via the @DataKinds@ extension.
data TypeKind =
    TyUnit
  | TyCtrl
  | (:->) TypeKind TypeKind
  | TyApp TypeKind TypeKind
  | Ty Symbol
  | TyTuple TypeKind TypeKind
  | TyVar Symbol
  | TyForAll Symbol TypeKind

-- | Statically typed representation of types. We use a GADT to parameterize the
-- type constructor 'Type' with phantom types of kind 'TypeKind'.
data Type :: TypeKind -> * where
  TypeUnit   :: Type TyUnit
  TypeCtrl   :: Type TyCtrl
  Type       :: KnownSymbol t              => Proxy t -> Type (Ty t)
  TypeFun    :: (GenType a, GenType b)     => Type a -> Type b -> Type (a :-> b)
  TypeApply  :: (GenType a, GenType b)     => Type a -> Type b -> Type (TyApp a b)
  TypeTuple  :: (GenType a, GenType b)     => Type a -> Type b -> Type (TyTuple a b)
  TypeVar    :: KnownSymbol t              => Proxy t -> Type (TyVar t)
  TypeForAll :: (KnownSymbol v, GenType t) => Proxy v -> Type t -> Type (TyForAll v t)
--------------------------------------------------------------------------------
deriving instance Eq (Type t)
--------------------------------------------------------------------------------
instance TestEquality Type where
  testEquality TypeUnit TypeUnit = Just Refl
  testEquality TypeCtrl TypeCtrl = Just Refl
  testEquality (Type a) (Type b) = do
    Refl <- sameSymbol a b
    return Refl
  testEquality (TypeFun a1 b1) (TypeFun a2 b2) = do
    Refl <- testEquality a1 a2
    Refl <- testEquality b1 b2
    return Refl
  testEquality (TypeApply a1 b1) (TypeApply a2 b2) = do
    Refl <- testEquality a1 a2
    Refl <- testEquality b1 b2
    return Refl
  testEquality (TypeTuple a1 b1) (TypeTuple a2 b2) = do
    Refl <- testEquality a1 a2
    Refl <- testEquality b1 b2
    return Refl
  testEquality (TypeForAll v1 t1) (TypeForAll v2 t2) = do
    Refl <- sameSymbol v1 v2
    Refl <- testEquality t1 t2
    return Refl
  testEquality (TypeVar a) (TypeVar b) = do
    Refl <- sameSymbol a b
    return Refl
  testEquality _ _ = Nothing
-------------------------------------------------------------------------------
instance Show (Type t) where
  showsPrec _ TypeUnit = showString "()"
  showsPrec _ TypeCtrl = showString "#"
  showsPrec _ (Type t) = showString (symbolVal t)
  showsPrec i (TypeFun a b) = showParen (i > 0) $ showsPrec 1 a . showString " → " . showsPrec 0 b
  showsPrec _ t@(TypeApply (TypeApply _ _) _)
    | Just (a,b) <- maybeTupleTypes t = showString "(" . shows a . showString "," . shows b . showString ")"
  showsPrec i t@(TypeApply a b)
    | Just a <- maybeListType t = showChar '[' . shows a . showChar ']'
    | otherwise = showParen (i > 9) $ showsPrec 9 a . showChar ' ' . showsPrec 10 b
  showsPrec _ (TypeTuple a b) = showString "(" . shows a . showString "," . shows b . showString ")"
  showsPrec _ (TypeVar v) = showString (symbolVal v)
  showsPrec _ (TypeForAll v t) = showString "forall " . showString (symbolVal v) . showString ". " . shows t
-------------------------------------------------------------------------------
-- | For every 'Typed' data type we can return a representation of its type.
class Typed (f :: TypeKind -> *) where
  typeOf :: f t -> Type t

instance Typed Type where
  typeOf t = t
--------------------------------------------------------------------------------
type TypeUnit      = TyUnit
type TypeCtrl      = TyCtrl
type PrimString    = Ty "Addr#"
type PrimChar      = Ty "Char#"
type PrimInt       = Ty "Int#"
type PrimDouble    = Ty "Double#"
type PrimFloat     = Ty "Float#"

type TypeString    = TyApp (Ty "[]") TypeChar
type TypeChar      = Ty "Char"
type TypeInt       = Ty "Int"
type TypeInteger   = Ty "Integer"
type TypeDouble    = Ty "Double"
type TypeFloat     = Ty "Float"

typePrimString :: Type PrimString
typePrimString = Type Proxy

typePrimChar :: Type PrimChar
typePrimChar = Type Proxy

typePrimInt :: Type PrimInt
typePrimInt = Type Proxy

typeInteger :: Type TypeInteger
typeInteger = Type Proxy

typePrimDouble :: Type PrimDouble
typePrimDouble = Type Proxy

typePrimFloat :: Type PrimFloat
typePrimFloat = Type Proxy

typeUnit :: Type TypeUnit
typeUnit = TypeUnit

typeInt :: Type TypeInt
typeInt = Type Proxy

typeChar :: Type TypeChar
typeChar = Type Proxy

typeFloat :: Type TypeFloat
typeFloat = Type Proxy

typeDouble :: Type TypeDouble
typeDouble = Type Proxy

typeString :: Type TypeString
typeString = TypeApply (Type Proxy) typeChar
--------------------------------------------------------------------------------
maybeTupleTypes :: Type (TyApp (TyApp c a) b) -> Maybe (Type a, Type b)
maybeTupleTypes t = case t of
  TypeApply (TypeApply c@(Type Proxy) a) b ->
    case testEquality c (Type Proxy :: Type (Ty "(,)")) of
      Just Refl -> Just (a,b)
      Nothing   -> Nothing
  _ -> Nothing
--------------------------------------------------------------------------------
maybeListType :: Type (TyApp c a) -> Maybe (Type a)
maybeListType t = case t of
  TypeApply c@(Type Proxy) a ->
    case testEquality c (Type Proxy :: Type (Ty "[]")) of
      Just Refl -> Just a
      Nothing   -> Nothing
  _ -> Nothing
--------------------------------------------------------------------------------
uncurryType :: Type (a :-> (b :-> c)) -> Type (TyTuple a b :-> c)
uncurryType (TypeFun a (TypeFun b c)) = TypeFun (TypeTuple a b) c
--------------------------------------------------------------------------------
-- | Type class for generating type representations from type information.
-- Instances are available for all types @t@ of kind @TypeKind@.
class GenType (t :: TypeKind) where
  genType :: Type t

instance GenType TyUnit where
  genType = TypeUnit

instance GenType TyCtrl where
  genType = TypeCtrl

instance KnownSymbol t => GenType (Ty t) where
  genType = Type Proxy

instance KnownSymbol t => GenType (TyVar t) where
  genType = TypeVar Proxy

instance (GenType a, GenType b) => GenType (a :-> b) where
  genType = TypeFun genType genType

instance (GenType a, GenType b) => GenType (TyApp a b) where
  genType = TypeApply genType genType

instance (GenType a, GenType b) => GenType (TyTuple a b) where
  genType = TypeTuple genType genType

instance (KnownSymbol v, GenType t) => GenType (TyForAll v t) where
  genType = TypeForAll Proxy genType
--------------------------------------------------------------------------------
