{-# LANGUAGE MultiParamTypeClasses #-}
{-# LANGUAGE FlexibleInstances #-}
{-# LANGUAGE FunctionalDependencies #-}

module Flux.Typed.Unification (
    -- * Substitutions
    Subst,
    emptySubst,
    (|->),
    (>+),
    lookupSubst,
    filterSubst,
    -- * Unification
    Unifiable(..),
    unifylist,
    Occurrence(..)
  )
  where
--------------------------------------------------------------------------------
import qualified Data.Map as Map
import Data.Monoid(Monoid(..))
import Data.List(intercalate, nub, intersect)
--------------------------------------------------------------------------------
type Map = Map.Map
--------------------------------------------------------------------------------
-- | substitute variables of type @v@ within terms of type @t@
newtype Subst v t = Subst (Map v t)
  deriving Eq

instance Occurrence v t => Occurrence v (Subst v t) where
  freeVars (Subst m) = nub (Map.keys m ++ concatMap freeVars (Map.elems m))

instance (Show v, Show t) => Show (Subst v t) where
  show (Subst s) =
    "{" ++
    intercalate "," [ "[" ++ show v ++ " |-> " ++ show t ++ "]" | (v,t) <- Map.assocs s ]
    ++ "}"

-- | the empty substitution
emptySubst :: (Ord v) => Subst v t
emptySubst = Subst (Map.empty)

-- | a single substitution/assignment of a variable
(|->) :: (Ord v) => v -> t -> Subst v t
x |-> p = Subst (Map.singleton x p)

-- | compose two substitutions, i.e. returns a substitution that has the result
-- of applying both substitutions after each other
(>+) :: Unifiable v t => Subst v t -> Subst v t -> Subst v t
-- note that union is left-biased, i.e. the effects of the first substitution
-- take precedence
(Subst s1) >+ s@(Subst s2) = Subst ((Map.map (s@@) s1) `Map.union` s2)

-- | return the term that a variable should be substituted with
lookupSubst :: (Ord v) => v -> Subst v t -> Maybe t
lookupSubst v (Subst s) = Map.lookup v s

filterSubst :: (v -> t -> Bool) -> Subst v t -> Subst v t
filterSubst p (Subst m) = Subst (Map.filterWithKey p m)

mapTerms :: (t -> t') -> Subst v t -> Subst v t'
mapTerms f (Subst m) = Subst (Map.map f m)
--------------------------------------------------------------------------------
class (Ord v) => Unifiable v t | t -> v where
  -- | apply substitution to a term
  (@@) :: Subst v t -> t -> t

  -- | unify two terms and return a list of subsitutions
  unify :: t -> t -> [Subst v t]

-- | unify two lists of terms
unifylist :: Unifiable v t => [t] -> [t] -> [Subst v t]
unifylist = unifylist' (Subst Map.empty)
  where
    unifylist' s [] []             = [s]
    unifylist' s (p1:ps1) (p2:ps2) =
        [s'' | s'  <- unify p1 p2
             , s'' <- unifylist' (s >+ s')
                                 [ s' @@ p | p <- ps1 ]
                                 [ s' @@ p | p <- ps2 ]
        ]
    unifylist' _ _ _ = []
--------------------------------------------------------------------------------
-- | Type class for expressions that contain certain kinds of variables. 
class (Eq v) => Occurrence v expr | expr -> v where
  freeVars :: expr -> [v]
  occursIn :: v -> expr -> Bool
  occursIn x e = x `elem` freeVars e

-- | Check whether two expressions share any variables
sharesVarsWith :: (Occurrence v e1, Occurrence v e2) => e1 -> e2 -> Bool
sharesVarsWith x y = (not . null) (freeVars x `intersect` freeVars y)
--------------------------------------------------------------------------------
