{-# LANGUAGE TypeOperators #-}
{-# LANGUAGE GADTs #-}
--------------------------------------------------------------------------------
module Main where

import Flux.Typed.CoreToExpr
import Flux.Typed.Cat
import Flux.Typed.GraphViz
import Flux.Typed.Expr
import Flux.Typed.Var
import Flux.Typed.Literal
import Flux.Typed.Pattern
import Flux.Typed.Type
import Flux.Typed.Boxed
import Flux.Typed.FluxGraph
import Flux.Typed.ReduceGraph
import Flux.Typed.Graph

import System.Directory(doesFileExist, createDirectoryIfMissing)
import System.Environment(getArgs, getProgName)
import System.Process(system)
import Control.Monad.IO.Class(liftIO)
import Control.Monad

import GHC hiding (Type, Pat)
import GHC.Paths(libdir)
import CoreSyn hiding (Expr(..))
import DynFlags
import Outputable
import Type hiding (Type)
import Var hiding (Var)
import HscTypes
import Panic
--------------------------------------------------------------------------------
outputFolder :: String
outputFolder = "output"
--------------------------------------------------------------------------------
escape :: String -> String
escape = concatMap esc
  where
    esc '$' = "\\$"
    esc c   = [c]
--------------------------------------------------------------------------------
main :: IO ()
main = do
    prog <- getProgName
    args <- getArgs
    case args of
      [operation, filename] -> do
        exists <- doesFileExist filename
        if exists
        then process operation filename
        else putStrLn $ "File not found: " ++ filename

      _ -> do putStrLn ("usage: " ++ prog ++ " OPERATION FILEPATH")
              putStrLn ("only currently implemented OPERATION is \"functions\"")
              putStrLn ("FILEPATH must be the path to a valid Haskell module file (*.hs)")


  where
    process :: String -> String -> IO ()
    process operation filename = defaultErrorHandler defaultFatalMessager defaultFlushOut $ do
      runGhc (Just libdir) $ do
        dflags <- getDynFlags
        let dflags' = dflags {
              ghcMode = CompManager,
              ghcLink = NoLink,
              hscTarget = HscNothing,
              optLevel = 0,
              simplPhases = 0
            }
        setSessionDynFlags dflags'
        t <- guessTarget filename Nothing 
        setTargets [t]
        Succeeded <- handleSourceError (\e -> liftIO $ putStrLn "Source Error" >> return Failed) (load LoadAllTargets)
        moduleGraph <- getModuleGraph
        forM_ moduleGraph $ \ms -> do
            cm <- parseModule ms >>= typecheckModule >>= desugarModule
            forM_ (mg_binds $ coreModule cm) $ \cb ->
              case cb of
                NonRec v e -> when (isExportedId v) (translate dflags operation v e)
                Rec bs     -> mapM_ (uncurry (\v e -> when (isExportedId v) (translate dflags operation v e))) bs
      return ()
--------------------------------------------------------------------------------
translate :: (GhcMonad m) => DynFlags -> String -> Type.Var -> CoreExpr -> m ()
translate dflags operation v e = do
    dflags <- getSessionDynFlags
    liftIO $ do
      putStrLn $ showSDocDebug dflags (ppr e)
      case translateCore dflags e of
          Left msg -> putStrLn $ "Translating core expression failed for: \n" ++ showSDoc dflags (ppr e) ++ "\n\n" ++ msg ++ "\n"
          Right box
            | Boxed expr  <- etaExpand box -> do
              case operation of
                "functions" |
                  TypeFun _ _ <- typeOf expr -> mkOutput (varToString v) (Just expr) (exprToCat expr)
                _                            -> error $ "unknown operation: '" ++ operation ++ "'"
--------------------------------------------------------------------------------
mkOutput :: (GenType b, GenType c) => String -> Maybe (Expr a) -> (b ~> c) -> IO ()
mkOutput name maybeExpr cat = do
    case maybeExpr of 
      Just expr -> do
        putStrLn "-----"
        putStr "Type: "
        print (typeOf expr)

        putStrLn "-----"
        putStr "Expr: "
        print expr
        putStrLn "-----"
      _ -> return ()

    putStr "Category: "
    print cat
    putStrLn "-----"
    let graph  = catToGraph cat
    outputGraph name graph
--------------------------------------------------------------------------------
outputGraph :: String -> Graph -> IO ()
outputGraph name graph = do
    let output = graphToDot graph
    putStr "Graph: "
    print graph
    putStrLn "-----"
    createDirectoryIfMissing True outputFolder
    writeFile outFile output
    _ <- system $ "dot " ++ escape outFile ++ " -Tsvg >" ++ svgFile
    return ()
  where
    outFile = outputFolder ++ "/" ++ name ++ ".txt"
    svgFile = outputFolder ++ "/" ++ name ++ ".svg"
--------------------------------------------------------------------------------
