/*
    This file is part of SGDAE, a software to numerically solve differential
    algebraic equations using a steepest descent method based on Sobolev 
    gradients.
    You are welcome to contact the authors via e-mail:
        <manfred-sauter [at] gmx [dot] de>
        <robin.nittka [at] gmx [dot] de>

    Copyright 2005-2008 Manfred Sauter, Robin Nittka.

    SGDAE is free software distributed under the terms of the revised BSD 
    license as illustrated on <http://creativecommons.org/licenses/BSD/>.
    For details consult the accompanying LICENSE.txt file.

    $Id: $
*/


#ifndef __LOGGING_HEADER__
#define __LOGGING_HEADER__


#include <cassert>
#include <streambuf>
#include <iostream>
#include <vector>
#include <string>

// TODO: What about supporting boost::iostreams?
// http://www.boost.org/libs/iostreams/doc/index.html
namespace logging
{
using std::endl;
    

typedef std::vector<std::ostream> tVOS;
typedef std::vector<std::string> tVS;


template <class CT, class Traits = std::char_traits<CT> >
class basic_null_streambuf : public std::basic_streambuf<CT, Traits>
{
    typedef typename Traits::int_type int_type;

protected:
	int_type overflow(int_type c)
    {
        return Traits::not_eof(c);
    }
	std::streamsize xsputn(const CT *s, std::streamsize n)
    {
        return n;
    }

public:
    basic_null_streambuf() {}
};

template <class CT, class Traits = std::char_traits<CT> >
class basic_null_ostream: public std::basic_ostream<CT, Traits> {
public:
    basic_null_ostream() : std::basic_ios<CT, Traits>(&m_sbuf), std::basic_ostream<CT, Traits>(&m_sbuf)
    {
        init(&m_sbuf);
    }

private:
    basic_null_streambuf<CT, Traits> m_sbuf;
};

typedef basic_null_ostream<char> tNull_ostream;


// Slight adaption from the code of "Tee streams in C++" on
// http://www.cs.technion.ac.il/~imaman/programs/programs.html
template<typename CT, typename Traits = std::char_traits<CT> >
class basic_tee_stream : std::basic_ostream<CT,Traits>
{
    typedef std::basic_ostream<CT,Traits> SuperType;
public:
    basic_tee_stream(std::ostream &o1, std::ostream &o2) : SuperType(o1.rdbuf()), o1_(o1), o2_(o2) { }

    basic_tee_stream &operator <<(SuperType &(*manip)(SuperType &))
    {
        o1_ << manip;
        o2_ << manip;
        return *this;
    }

    template<typename T> basic_tee_stream &operator <<(const T& t)
    {
        o1_ << t;
        o2_ << t;
        return *this;
    }

private:
    std::ostream &o1_;
    std::ostream &o2_;
};

// Adapted version from http://www.ddj.com/cpp/184403387
template <class CT, class Traits = std::char_traits<CT> >
class basic_tee_streambuf : public std::basic_streambuf<CT, Traits>
{
    typedef typename std::basic_streambuf<CT, Traits> sb_type;
    typedef typename Traits::int_type int_type;

public:
    basic_tee_streambuf(sb_type *s1, sb_type *s2) : buf1(s1), buf2(s2) {}

    ~basic_tee_streambuf()
    {
        sync();
    }

protected:
    std::streamsize xsputn(const CT *s, std::streamsize n)
    {
        std::streamsize res1 = buf1->sputn(s, n);
        std::streamsize res2 = buf2->sputn(s, n);
        return std::min(res1, res2);
    }

    int_type overflow(int_type c)
    {
        buf1->sputc(c);
        buf2->sputc(c);
        return c;
    }
    
    int sync()
    {
        if (-1 == buf1->pubsync()) return -1;
        return buf2->pubsync();
    }

private:
    sb_type *buf1;
    sb_type *buf2;
};

typedef basic_tee_streambuf<char> tee_streambuf;

enum tLogType
{
	DEBUG = 0,
	INFO,
	WARNING,
	ERROR,
	FATAL,
	LOG_TYPE_END,
	MASK_ALL,
	LOG_MASK_END	
};

class tLogger : public std::ostream
{
private:	
	std::string name[LOG_TYPE_END];
	std::ostream *os[LOG_TYPE_END];
	tVS scope_stack;

	static tNull_ostream null_os;
	static tLogger *logger;
	
	tLogger();
	void pushScope(const std::string &scope);
	void popScope();
public:
	static tLogger &getLogger()
    {
        if (logger == NULL) logger = new tLogger;
        return *logger;
    }

	void tie(const tLogType type, std::ostream *out);
	std::ostream *get_tie(const tLogType type);
    void setName(const tLogType, const std::string &);
	bool isLogTypeLogged(const tLogType type) const
    {
        assert(type < LOG_TYPE_END);
        return os[type] != NULL;
    }

	std::ostream &operator ()(const tLogType type = INFO);

	friend class tLogScope;
};



class tLogScope
{
public:
	tLogScope(const std::string &env)
	{
		this->getLogger()(DEBUG) << "entering " << env << endl;
		tLogger::getLogger().pushScope(env);
	}

	virtual ~tLogScope()
	{
		tLogger::getLogger().popScope();
		this->getLogger()(DEBUG) << "left scope" << endl;
	}

	bool isLogTypeLogged(const tLogType type) const
    {
        return tLogger::getLogger().isLogTypeLogged(type);
    }

	tLogger &getLogger() const
	{
		return tLogger::getLogger();
	}

	std::ostream &operator ()(const tLogType type) const
	{
		return this->getLogger()(type);
	}
};

#define LOG(type)           (tLogger::getLogger().isLogTypeLogged(type)) && tLogger::getLogger()(type)
#define SCOPE(name)         tLogScope scope(name);

};

#endif
