/*
    This file is part of SGDAE, a software to numerically solve differential
    algebraic equations using a steepest descent method based on Sobolev 
    gradients.
    You are welcome to contact the authors via e-mail:
        <manfred-sauter [at] gmx [dot] de>
        <robin.nittka [at] gmx [dot] de>

    Copyright 2005-2008 Manfred Sauter, Robin Nittka.

    SGDAE is free software distributed under the terms of the revised BSD 
    license as illustrated on <http://creativecommons.org/licenses/BSD/>.
    For details consult the accompanying LICENSE.txt file.

    $Id: main.cc 413 2008-02-11 14:41:09Z msauter $
*/


// STL headers.
#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <string>
#include <vector>
#include <set>
#include <cmath>
#include <ctime>
#include <cassert>

#include "logging.h"
#include "utility.hxx"
#include "type.hxx"
#include "solvers.hxx"
#include "problems.hxx"
#include "descent.hxx"




void testcase_numerically_difficult()
{
    using namespace std;
    using namespace problems;
    using namespace steepest_descent;

    const int N = 1000;
    const Real lambda = 1.e-3;
    const Real damping = 0.85;
    const SobolevNorm style = NORM_GRAPH;
    const int MAX_STEPS = 10000;
    const Real eta = -0.8;

	typedef vector<int> tVI;
	tVI steps;
	for (int i=0; i<100; i++) steps.push_back(1);
	for (int i=0; i<40; i++) steps.push_back(10);
	for (int i=0; i<30; i++) steps.push_back(50);
	for (int i=0; i<8; i++) steps.push_back(1000);

    ProblemDifficult problem(eta);
    EmptySupplementaryConditions<ProblemDifficult> supp(problem, N);
    SteepestDescent<EmptySupplementaryConditions<ProblemDifficult> > sd(supp);

    int plot_cnt = 0;
    for (int k=0; k<=sd.N; k++)
    {
        DenseVector tmp(sd.problem.n);
        init(tmp) = 2., 2.;
        sd.set_current_estimate(k, tmp);
    }
    ofstream data("data.dat");
    data.precision(16);
    data.setf(ios::fixed);
    gnuplot_output(sd, data);
    data << endl << endl;
    ++plot_cnt;

    {
        Real avg_error = 0.;
        Real max_error = 0.;
        for (int k=0; k<=sd.N; k++)
        {
            DenseVector tmp(sd.problem.n);
            sd.get_current_estimate(k, tmp);

            const Real t = k*sd.delta;
            DenseVector sol(sd.problem.n);
            init(sol) = (1.-eta*t)*exp(-t), exp(-t);
            tmp -= sol;

            max_error = max(max_error, ublas::norm_inf(tmp));
            avg_error += ublas::inner_prod(tmp, tmp);
        }
        avg_error *= sd.problem.T / (sd.N+1.); 
        sd.init_descent(style, sd.u);
        LOG(INFO) << "START " << sd.eval_psi(sd.u) << " " << avg_error << " " << max_error << endl;
    }

    for (tVI::size_type j=0; j<steps.size(); j++)
    {
        bool break_ = false;
        try
        {
            sd.descent(style, steps[j], damping, lambda);
            gnuplot_output(sd, data);
            data << endl << endl;
            ++plot_cnt;
        }
        catch (std::exception &e)
        {
            std::cerr << e.what() << endl;
            LOG(INFO) << e.what() << endl;
            break_ = true;
        }

        Real avg_error = 0.;
        Real max_error = 0.;
        for (int k=0; k<=sd.N; k++)
        {
            DenseVector tmp(sd.problem.n);
            sd.get_current_estimate(k, tmp);

            const Real t = k*sd.delta;
            DenseVector sol(sd.problem.n);
            init(sol) = (1.-eta*t)*exp(-t), exp(-t);
            tmp -= sol;

            max_error = max(max_error, ublas::norm_inf(tmp));
            avg_error += ublas::inner_prod(tmp, tmp);
        }
        avg_error *= sd.problem.T / (sd.N+1.); 
		LOG(INFO) << sd.step << " " << sd.eval_psi(sd.u) << " " << avg_error << " " << max_error << endl;

        if (sd.step >= MAX_STEPS || break_) break;
    }

    LOG(INFO) << "Number of generated plots: " << plot_cnt << endl;
}


void testcase_singularity()
{
    using namespace std;
    using namespace problems;
    using namespace steepest_descent;

    using utility::sqr;

//		stringstream ssdata;
//		ssdata.setf(ios::scientific);
//		ssdata << "singular_data_" << size[isize] << SOBOLEV_NORM_NAME[style[istyle]] << lambda[ilambda] << ".dat";

    const int N = 10000;
    const Real lambda = 1.;
    const Real damping = 0.85;
    const SobolevNorm style = NORM_WEIGHTED;
    const int MAX_STEPS = 1000;

    
    LOG(INFO) << "grid size: " << N << endl;
    LOG(INFO) << "lambda: " << lambda << endl;
    LOG(INFO) << "damping: " << damping << endl;
    LOG(INFO) << "gradient: " << SOBOLEV_NORM_NAME[style] << endl;
      
	typedef vector<int> tVI;
	tVI steps;
	for (int i=0; i<100; i++) steps.push_back(1);
	for (int i=0; i<40; i++) steps.push_back(10);
	for (int i=0; i<30; i++) steps.push_back(50);
	for (int i=0; i<8; i++) steps.push_back(1000);

    ProblemSingularity2 problem;
    DirichletSupplementaryConditions<ProblemSingularity2> supp(problem, N);
    supp.fix_value(0, N);

    SteepestDescent<DirichletSupplementaryConditions<ProblemSingularity2> > sd(supp);

    for (int k=0; k<=sd.N; k++)
    {
        const Real t = k*sd.delta;
        DenseVector tmp(sd.problem.n);
        tmp(0) = t;
//        tmp(0) = t*t/(2.-t); // exact solution
        sd.set_current_estimate(k, tmp);
    }

    {
        Real avg_error = 0.;
        Real max_error = 0.;
        for (int k=0; k<=sd.N; k++)
        {
            DenseVector tmp(sd.problem.n);
            sd.get_current_estimate(k, tmp);

            const Real t = k*sd.delta;
            tmp(0) -= t*t/(2.-t);

            max_error = max(max_error, fabs(tmp(0)));
            avg_error += sqr(tmp(0));
        }
        avg_error *= sd.problem.T / (sd.N+1.); 
        LOG(INFO) << "START " << sd.eval_psi(sd.u) << " " << avg_error << " " << max_error << endl;
    }

    for (tVI::size_type j=0; j<steps.size(); j++)
    {
        bool break_ = false;
        try
        {
            sd.descent(style, steps[j], damping, lambda);
        }
        catch (std::exception &e)
        {
            std::cerr << e.what() << endl;
            LOG(INFO) << e.what() << endl;
            break_ = true;
        }

        Real avg_error = 0.;
        Real max_error = 0.;
        for (int k=0; k<=sd.N; k++)
        {
            DenseVector tmp(sd.problem.n);
            sd.get_current_estimate(k, tmp);

            const Real t = k*sd.delta;
            tmp(0) -= t*t/(2.-t);

            max_error = max(max_error, fabs(tmp(0)));
            avg_error += sqr(tmp(0));
        }
        avg_error *= sd.problem.T / (sd.N+1.); 
		LOG(INFO) << sd.step << " " << sd.eval_psi(sd.u) << " " << avg_error << " " << max_error << endl;

        if (sd.step >= MAX_STEPS || break_) break;
    }
}

void testcase_figure_eight()
{
	using namespace std;
    using namespace problems;
    using namespace steepest_descent;

    using utility::rand_uniform;

    ProblemFigureEight problem;
    EmptySupplementaryConditions<ProblemFigureEight> supp(problem, 300);
    SteepestDescent<EmptySupplementaryConditions<ProblemFigureEight> > sd(supp);

    ofstream data("figure_eight_dim.dat");
    data.precision(16);
    data.setf(ios::fixed);
    for (int i=0; i<10000; i++)
    {
        DenseVector tmp1(2), tmp2(2);
        init(tmp1) = rand_uniform(-2.,2.), rand_uniform(-2.,2.);
        init(tmp2) = rand_uniform(-2.,2.), rand_uniform(-2.,2.);

        for (int k=0; k<=sd.N; k++)
        {
            const Real h = k*sd.delta;
            const Real lambda = h/sd.problem.T;
            sd.set_current_estimate(k, (1.-lambda)*tmp1 + lambda*tmp2);
        }
        sd.descent(NORM_GRAPH, 30, 0.85, 1.e-5);

        DenseVector init(2);
        sd.get_current_estimate(0, init);
        data << init(0) << " " << init(1) << endl;
    }

}

void testcase_dim()
{
	using namespace std;
    using namespace problems;
    using namespace steepest_descent;

    using utility::rand_uniform;
/*
    ProblemLinearDim problem(ProblemLinearDim::INDEX0);
    EmptySupplementaryConditions<ProblemLinearDim> supp(problem, 1000);
    SteepestDescent<EmptySupplementaryConditions<ProblemLinearDim> > sd(supp);
*/
	ProblemLinearDimBig problem;
	EmptySupplementaryConditions<ProblemLinearDimBig> supp(problem, 100);
	SteepestDescent<EmptySupplementaryConditions<ProblemLinearDimBig> > sd(supp);

    ofstream full("data.dat");
    full.precision(25);
    full.setf(ios::fixed);

    ofstream data("linear_dim.dat");
    data.precision(25);
    data.setf(ios::fixed);
    for (int i=0; i<500; i++)
    {
        DenseVector tmp1(problem.n), tmp2(problem.n);
        for (int j=0; j<problem.n; j++)
        {
            tmp1(j) = rand_uniform(-2.,2.);
            tmp2(j) = rand_uniform(-2.,2.);
        }

        for (int k=0; k<=sd.N; k++)
        {
            const Real h = k*sd.delta;
            const Real lambda = h/sd.problem.T;
            sd.set_current_estimate(k, (1.-lambda)*tmp1 + lambda*tmp2);
        }
        sd.descent(NORM_H1, 2000, 0.85, 1.);
        sd.descent(NORM_GRAPH, 5000, 0.85, 1.e-8);

		gnuplot_output(sd, full);
        full << endl << endl;

        DenseVector init(problem.n);
		// TODO:
        sd.get_current_estimate(50, init);
        for (int j=0; j<problem.n; j++)
            data << init(j) << " ";
        data << endl;
    }
}

void testcase_nonunique()
{
	using namespace std;
    using namespace problems;
    using namespace steepest_descent;

    using utility::rand_uniform;

    ProblemNonunique problem;
    DirichletSupplementaryConditions<ProblemNonunique> supp(problem, 500);
	supp.fix_value(0,0);
	supp.fix_value(1,0);
    SteepestDescent<DirichletSupplementaryConditions<ProblemNonunique> > sd(supp);

    ofstream data("nonunique_dim.dat");
    data.precision(16);
    data.setf(ios::fixed);
    for (int i=0; i<100; i++)
    {
        DenseVector tmp1(2), tmp2(2);
        init(tmp1) = 0., 0.;
        init(tmp2) = rand_uniform(-2.,2.), rand_uniform(-2.,2.);

        for (int k=0; k<=sd.N; k++)
        {
            const Real h = k*sd.delta;
            const Real lambda = h/sd.problem.T;
            sd.set_current_estimate(k, (1.-lambda)*tmp1 + lambda*tmp2);
        }
        sd.descent(NORM_GRAPH, 1000, 0.85, 1.e-2);
		gnuplot_output(sd, data);
		// gnuplot: to seperate multiple plots in a single file
		data << endl << endl;
    }
}

void testcase_bvp()
{
	using namespace std;
    using namespace problems;
    using namespace steepest_descent;

    using utility::rand_uniform;
/*
	ProblemBVP problem(1.);
	DirichletSupplementaryConditions<ProblemBVP> supp(problem, 1000);
	supp.fix_value(0, 0);
	supp.fix_value(1, supp.N);
	supp.fix_value(2, 0);
    SteepestDescent<DirichletSupplementaryConditions<ProblemBVP> > sd(supp);


    ofstream data("bvp_nonunique.dat");
    data.precision(25);
    data.setf(ios::fixed);
    for (int i=0; i<100; i++)
    {
        DenseVector tmp1(4), tmp2(4);
        init(tmp1) = sin(0.), rand_uniform(-2.,2.), 1., rand_uniform(-20.,20.);
        init(tmp2) = rand_uniform(-2.,2.), sin(1.), rand_uniform(-2.,2.), rand_uniform(-20.,20.);
//       init(tmp1) = sin(0.), 2.000, 1., 0.000;
//       init(tmp2) = 3.000, sin(1.), 1.000, 0.000;

        for (int k=0; k<=sd.N; k++)
        {
            const Real h = k*sd.delta;
            const Real lambda = h/sd.problem.T;
            sd.set_current_estimate(k, (1.-lambda)*tmp1 + lambda*tmp2);
//            init(tmp1) = sin(h), sin(h), 1., 0.;
//            init(tmp1) = sin(h)+exp(h)-1., sin(h), exp(h), exp(h);
//            sd.set_current_estimate(k, tmp1);
        }

        try
        {
            sd.descent(NORM_GRAPH, 200, 0.85, 1e-6);
        }
        catch (std::exception &e)
        {
            LOG(INFO) << e.what() << endl;
        }
          
        // gnuplot: to seperate multiple plots in a single file accessible by the keyword index
        gnuplot_output(sd, data);
        data << endl << endl;
    }
*/
    

	ProblemBVPSimplified problem(1.);
	DirichletSupplementaryConditions<ProblemBVPSimplified> supp(problem, 100);
	supp.fix_value(0, 0);
    SteepestDescent<DirichletSupplementaryConditions<ProblemBVPSimplified> > sd(supp);

    ofstream data("bvp_nonunique_simple.dat");
    data.precision(25);
    data.setf(ios::fixed);
    for (int i=0; i<100; i++)
    {
        LOG(INFO) << "TRY NUMBER " << i+1 << endl;

        DenseVector tmp1(2), tmp2(2);
        init(tmp1) = sin(0.), rand_uniform(-10.,10.);
        init(tmp2) = rand_uniform(-2.,2.), rand_uniform(-10.,10.);

//        init(tmp1) = 0.00000000000000000000000000000000000, 6.23279519028290707183259655721485615;
//        init(tmp2) = 0.29218420972319725237298371212091297, -3.07779168065431729672809524345211685;

        LOG(INFO) << "tmp1: " << tmp1 << endl;
        LOG(INFO) << "tmp2: " << tmp2 << endl;


        for (int k=0; k<=sd.N; k++)
        {
            const Real h = k*sd.delta;
            const Real lambda = h/sd.problem.T;
            sd.set_current_estimate(k, (1.-lambda)*tmp1 + lambda*tmp2);
        }
        try
        {
            sd.descent(NORM_GRAPH, 80, 0.85, 1e-6);
//            sd.descent(NORM_H1, 10000, 0.85, 1.);
        }
        catch (std::exception &e)
        {
            LOG(INFO) << e.what() << endl;
        }
        gnuplot_output(sd, data);
        data << endl << endl;
    }
    

#if 0
	ProblemSimple problem;
	DirichletSupplementaryConditions<ProblemSimple> supp(problem, 1000);
    supp.fix_value(0, 10);
    supp.fix_value(0, 501);
    supp.fix_value(0, 990);
    SteepestDescent<DirichletSupplementaryConditions<ProblemSimple> > sd(supp);

    ofstream data("data.dat");
    data.precision(25);
    data.setf(ios::fixed);
 //   for (int i=0; i<100; i++)
    {
/*        
        for (int k=0; k<=10; k++)
        {
            DenseVector tmp1(1);
            tmp1(0) = 0;
            sd.set_current_estimate(k, tmp1);
            sd.set_current_estimate(1000-k, tmp1);
        }
        
        for (int k=11; k<=501; k++)
        {
            DenseVector tmp1(1);
            tmp1(0) = (k-10.)/491.;

            sd.set_current_estimate(k, tmp1);
        }

        for (int k=502; k<990; k++)
        {
            DenseVector tmp1(1);
            tmp1(0) = 1.-(k-501.)/489.;

            sd.set_current_estimate(k, tmp1);
        }
*/       
        
        for (int k=0; k<=100; k++)
        {
            DenseVector tmp1(1);
            tmp1(0) = 0.;
            sd.set_current_estimate(k, tmp1);
            sd.set_current_estimate(1000-k, tmp1);
        }
        for (int k=101; k<900; k++)
        {
            DenseVector tmp1(1);
            tmp1(0) = 1.;
            sd.set_current_estimate(k, tmp1);
        }

        try
        {
//            sd.descent(NORM_GRAPH, 80, 0.85, 1e-6);
            sd.descent(NORM_H1, 250, 0.85, 1.);
        }
        catch (std::exception &e)
        {
            LOG(INFO) << e.what() << endl;
        }
        gnuplot_output(sd, data);
        data << endl << endl;
    }
#endif    
}

void testcase_ascher_petzold()
{
	using namespace std;
    using namespace problems;
    using namespace steepest_descent;
/*
    {
    ProblemAscherPetzold problem;
    DirichletSupplementaryConditions<ProblemAscherPetzold> supp(problem, 300);
    for (int i=0; i<3; i++)
        supp.fix_value(i, 0);

    SteepestDescent<DirichletSupplementaryConditions<ProblemAscherPetzold> > sd(supp);
    
    for (int k=0; k<=sd.N; k++)
        sd.set_current_estimate(k, problem.initial());
   
    sd.descent(NORM_GRAPH, 30, 0.85, 1.e-5);
    ofstream data("data.dat");
    data.precision(16);
    data.setf(ios::fixed);
    gnuplot_output(sd, data);
    }
*/

    using utility::rand_uniform;

    ProblemAscherPetzold problem;
    EmptySupplementaryConditions<ProblemAscherPetzold> supp(problem, 300);
    SteepestDescent<EmptySupplementaryConditions<ProblemAscherPetzold> > sd(supp);

    ofstream data("ascher_petzold.dim.dat");
    data.precision(16);
    data.setf(ios::fixed);
    for (int i=0; i<10; i++)
    {
        DenseVector tmp1(3), tmp2(3);
        init(tmp1) = rand_uniform(-2.,2.), rand_uniform(-2.,2.), rand_uniform(-2.,2.);
        init(tmp2) = rand_uniform(-2.,2.), rand_uniform(-2.,2.), rand_uniform(-2.,2.);

        for (int k=0; k<=sd.N; k++)
        {
            const Real h = k*sd.delta;
            const Real lambda = h/sd.problem.T;
            sd.set_current_estimate(k, (1.-lambda)*tmp1 + lambda*tmp2);
        }
        sd.descent(NORM_GRAPH, 30, 0.85, 1.e-5);

        DenseVector init(3);
        sd.get_current_estimate(0, init);
        data << init(0) << " " << init(1) << " " << init(2) << endl;
    }

}


void testcase_andrews()
{
	using namespace std;
    using namespace problems;
    using namespace steepest_descent;

/*

    const int STEPS = 100;
    const Real Tstep = 3e-2/STEPS;
    ProblemAndrewsIndex1Small problem(Tstep);
    DirichletSupplementaryConditions<ProblemAndrewsIndex1Small> supp(problem, 5000);

    for (int i=0; i<20; i++)
        supp.fix_value(i, 0);

    SteepestDescent<DirichletSupplementaryConditions<ProblemAndrewsIndex1Small> > sd(supp);
   
    ofstream data("data.dat");
    data.precision(16);
    data.setf(ios::fixed);


	DenseVector next = problem.initial();
    for (int step=0; step<STEPS; step++)
    {
        LOG(INFO) << "Step " << step+1 << ":" << endl;
        for (int k=0; k<=sd.N; k++)
            sd.set_current_estimate(k, next);
        
        sd.descent(NORM_H1, 10, 0.85, 1);
        sd.descent(NORM_GRAPH, 20, 0.85, 1e-5);
        sd.get_current_estimate(sd.N, next);

        const Real Tcur = step * Tstep;
        gnuplot_output(sd, data, Tcur);
    }
*/  
   

/*
    ProblemAndrewsIndex1Small problem(Tstep);
    typedef vector<DenseVector> tVDV;
    tVDV prev;
    prev.push_back(problem.initial());
    for (int step=0; step<STEPS; step++)
    {
        LOG(INFO) << "Step " << step+1 << ":" << endl;
        ProblemAndrewsIndex1Small problem(Tstep*(step+1));
        DirichletSupplementaryConditions<ProblemAndrewsIndex1Small> supp(problem, 50*(step+1));

        for (int i=0; i<20; i++)
            supp.fix_value(i, 0);

        SteepestDescent<DirichletSupplementaryConditions<ProblemAndrewsIndex1Small> > sd(supp);
        for (int k=0; k<int(prev.size()); k++)
            sd.set_current_estimate(k, prev[k]);
        for (int k=prev.size(); k<=sd.N; k++)
            sd.set_current_estimate(k, prev.back());
        
        sd.descent(NORM_H1, 10, 0.85, 1);
        sd.descent(NORM_GRAPH, 20, 0.85, 1e-5);

        for (int k=50*step+1; k<=50*(step+1); k++) 
        {
            DenseVector next;
            sd.get_current_estimate(k, next);
            prev.push_back(next);
        }

        ofstream data("data.dat");
        data.precision(16);
        data.setf(ios::fixed);
        gnuplot_output(sd, data);
    }
*/
/*
    ProblemAndrewsIndex1Small problem(Tstep);
    DirichletSupplementaryConditions<ProblemAndrewsIndex1Small> supp(problem, 100);

    for (int i=0; i<7; i++)
        supp.fix_value(i, 0);

    SteepestDescent<DirichletSupplementaryConditions<ProblemAndrewsIndex1Small> > sd(supp);
   
    ofstream data("data.dat");
    data.precision(16);
    data.setf(ios::fixed);


    typedef vector<DenseVector> tVDV;
    tVDV prev;
    prev.push_back(problem.initial());

    for (int step=0; step<STEPS; step++)
    {
        LOG(INFO) << "Step " << step+1 << ":" << endl;
        for (int k=0; k<=sd.N; k++)
            sd.set_current_estimate(k, prev.back());
        
        sd.descent(NORM_H1, 5, 0.85, 1);
        sd.descent(NORM_GRAPH, 20, 0.85, 1e-5);
//        sd.get_current_estimate(sd.N, next);

		for (int k=1; k<=sd.N; k++)
		{
			DenseVector tmp;
			sd.get_current_estimate(k, tmp);
			prev.push_back(tmp);
		}

        const Real Tcur = step * Tstep;
        gnuplot_output(sd, data, Tcur);


		if (step > 2)
		{
			// refine
			LOG(INFO) << "Refinement Step " << step+1 << endl;
			ProblemAndrewsIndex1Small problem2(Tstep*4);
			DirichletSupplementaryConditions<ProblemAndrewsIndex1Small> supp2(problem, 100*4);
			for (int i=0; i<7; i++)
				supp2.fix_value(i, 0);

			SteepestDescent<DirichletSupplementaryConditions<ProblemAndrewsIndex1Small> > sd2(supp2);
			for (int k=0; k<=sd2.N; k++)
				sd2.set_current_estimate(k, prev[prev.size()-400+k-1]);

			sd2.descent(NORM_H1, 5, 0.85, 1);
			sd2.descent(NORM_GRAPH, 20, 0.85, 1e-5);

			DenseVector tmp;
	        sd2.get_current_estimate(sd2.N, tmp);
			prev.pop_back();
			prev.push_back(tmp);
		}
    }
    */

    ProblemAndrewsIndex3 problem;
    DirichletSupplementaryConditions<ProblemAndrewsIndex3> supp(problem, 10000);

    for (int i=0; i<27; i++)
        supp.fix_value(i, 0);

    SteepestDescent<DirichletSupplementaryConditions<ProblemAndrewsIndex3> > sd(supp);
    
    ifstream input("../input/sol.andrews.10000.dat");
    gnuplot_input(sd, input);

    using utility::rand_uniform;
    DenseVector tmp1(problem.n), tmp2(problem.n);
    for (int i=0; i<27; i++) 
    {    
        tmp1(i) = 0.;
        tmp2(i) = 0.02*rand_uniform(-2.,2.);
    }    

    for (int k=0; k<=sd.N; k++) 
    {    
        const Real h = k*sd.delta;
        const Real lambda = h/problem.T;

        DenseVector tmp(problem.n);
        sd.get_current_estimate(k, tmp);
        sd.set_current_estimate(k, tmp + (1.-lambda)*tmp1 + lambda*tmp2);
    }    


/*    
    for (int k=0; k<=sd.N; k++)
    {
        const Real h = k * sd.delta;
	    sd.set_current_estimate(k, problem.initial());
    }
*/    
    sd.descent(NORM_H1, 10, 0.85, 1);
    sd.descent(NORM_GRAPH, 20, 0.85, 1e-8);
    ofstream data("data.dat");
    data.precision(25);
    data.setf(ios::fixed);
    gnuplot_output(sd, data);
}

void testcase_lorenz_attractor()
{
    using namespace std;
    using namespace problems;
    using namespace steepest_descent;

    ProblemLorenz problem(800.);
    EmptySupplementaryConditions<ProblemLorenz> supp(problem, 160000);
//    for (int i=0; i<3; i++)
//       supp.fix_value(i, 0);
    SteepestDescent<EmptySupplementaryConditions<ProblemLorenz> > sd(supp);

	ifstream input("data.dat");
    gnuplot_input(sd, input);

//    ifstream input("input/lorenz.dat");
//    ifstream input("backup/lorenz3/data.dat");
//	ifstream input("input/butterfly.dat");
//    gnuplot_input(sd, input);
/*
	for (int k=0; k<=sd.N; k++)
	{
		assert(sd.problem.n == 3);
		DenseVector tmp(3);
//		init(tmp) = sqrt(72.), sqrt(72.), 27.;
		init(tmp) = 10.,12.,14.;
//		init(tmp) = 3.,15.,1.;
//		init(tmp) = 2.5450919789293978, 5.7573376958333045, -2.6939528375278199;
//		init(tmp) = 10.0094834215091, 7.87724239501250, 26.3369924246729;		
		sd.set_current_estimate(k, tmp);

	}
*/	
	
//	sd.descent(NORM_H1, 40, 0.85, 1.);
	sd.descent(NORM_GRAPH, 40, 0.85, 1.e-4);

    ofstream data("data.dat");
    data.precision(16);
    data.setf(ios::fixed);
	gnuplot_output(sd, data);
}

void testcase_scaling()
{
    using namespace std;
    using namespace problems;
    using namespace steepest_descent;
    using utility::rand_uniform;

    ofstream out("data.dat");
    ProblemScaling problem;
    DirichletSupplementaryConditions<ProblemScaling> supp(problem, 1000);
    supp.fix_value(1,0);
    SteepestDescent<DirichletSupplementaryConditions<ProblemScaling> > sd(supp);

    for (int k=0; k<=sd.N; k++)
    {
//      const Real t = k*sd.delta;
        DenseVector tmp(sd.problem.n);
        init(tmp) = 1., 1.;   
//      init(tmp) = sin(t), exp(7.*sin(t));
        sd.set_current_estimate(k, tmp);
    }
    try
    {
        sd.descent(NORM_GRAPH, 100, 0.85, 1.e-5);
    }
    catch (std::exception &e)
    {
        LOG(INFO) << e.what() << endl;
    }
    gnuplot_output(sd, out);
    out << endl << endl;
}


void testcase_sonic()
{
    using namespace std;
    using namespace problems;
    using namespace steepest_descent;

    ProblemSonic problem;
    DirichletSupplementaryConditions<ProblemSonic> supp(problem, 1000);
    supp.fix_value(ProblemSonic::RHO, 0);
    supp.fix_value(ProblemSonic::RHO, 515);


    SteepestDescent<DirichletSupplementaryConditions<ProblemSonic> > sd(supp);

    for (int k=0; k<=sd.N; k++)
    {
        DenseVector tmp(3);
        tmp(ProblemSonic::PHI) = 1.;
        tmp(ProblemSonic::E) = 1.;
        tmp(ProblemSonic::RHO) = problem.rho_bar;

        sd.set_current_estimate(k, tmp);
    }

    sd.descent(NORM_GRAPH, 100, 0.85, 1.);

    ofstream data("data.dat");
    data.precision(16);
    data.setf(ios::fixed);
    gnuplot_output(sd, data);
}

void testcase_chemakzo()
{
    using namespace std;
    using namespace problems;
    using namespace steepest_descent;

/*
    ProblemChemakzo problem;
    DirichletSupplementaryConditions<ProblemChemakzo> supp(problem, 10000);
    for (int i=0; i<problem.n; i++)
        supp.fix_value(i, 0);

    SteepestDescent<DirichletSupplementaryConditions<ProblemChemakzo> > sd(supp);
    for (int i=0; i<=sd.N; i++)
        sd.set_current_estimate(i, problem.initial());

    sd.descent(NORM_GRAPH, 80, 0.85, 1e-5);
    ofstream data("data.dat");
    data.precision(16);
    data.setf(ios::fixed);
    gnuplot_output(sd, data);
*/        


    // T = 0.05
    ProblemChemakzo problem(0.05);
    SupplementaryConditions<ProblemChemakzo> supp(problem, 100);
    for (int i=0; i<problem.n; i++)
        supp.fix_value(i, 0);
    SteepestDescent<SupplementaryConditions<ProblemChemakzo> > sd(supp);

    DenseVector next = problem.initial();
    for (int j=0; j<180*2*10; j++)
    {
        cout << j*0.05 << ": " << next << endl;
        for (int i=0; i<=sd.N; i++)
            sd.set_current_estimate(i, next);
        sd.descent(NORM_GRAPH, 100, 1.0);

        sd.get_current_estimate(sd.N, next);
    }
    gnuplot_output(sd, cout);
}

int main()
{
	using namespace std;
    using namespace logging;

    utility::srand(time(NULL));


    cout.precision(35);
    cout.setf(ios::fixed);

    // Initialization of the logging.
    ofstream logfile("output.log");
    tee_streambuf logbuf(cout.rdbuf(), logfile.rdbuf());
    tee_streambuf errbuf(cerr.rdbuf(), logfile.rdbuf());
    ostream logstream(&logbuf);
    ostream errstream(&errbuf);

    logfile.precision(35);
    logfile.setf(ios::fixed);
    logstream.precision(35);
    logstream.setf(ios::fixed);
    errstream.precision(35);
    errstream.setf(ios::fixed);

    { 
        tLogger &log = tLogger::getLogger();
        log.tie(MASK_ALL, &errstream);
        log.tie(DEBUG, NULL /*&logfile*/);
        log.tie(INFO, &logstream);
    }

	boost::timer timer;

	try
	{
//		testcase_figure_eight();	// DONE
//		testcase_nonunique();		// DONE
//		testcase_singularity();     // DONE
//      testcase_numerically_difficult(); // DONE
  		testcase_bvp();             // DONE
//      testcase_dim();             // DONE
//      testcase_scaling();         // DONE

//      testcase_andrews();


//      testcase_chemakzo();
//      testcase_sonic();
//		testcase_lorenz_attractor();
//      testcase_ascher_petzold();

	}
	catch (const char *e)
	{
		LOG(ERROR) << e << endl;
    }

	LOG(INFO) << "Overall timing: " << timer.elapsed() << " sec" << endl;
	return 0;
}


