/*
    This file is part of SGDAE, a software to numerically solve differential
    algebraic equations using a steepest descent method based on Sobolev 
    gradients.
    You are welcome to contact the authors via e-mail:
        <manfred-sauter [at] gmx [dot] de>
        <robin.nittka [at] gmx [dot] de>

    Copyright 2005-2008 Manfred Sauter, Robin Nittka.

    SGDAE is free software distributed under the terms of the revised BSD 
    license as illustrated on <http://creativecommons.org/licenses/BSD/>.
    For details consult the accompanying LICENSE.txt file.

    $Id: $
*/


#ifndef __PROBLEMS_HXX__
#define __PROBLEMS_HXX__
namespace problems {
using boost::assign::init;

struct problem_linear {};
struct problem_linear_constant {};
struct problem_nonlinear {};


struct ProblemBase
{
	typedef DenseMatrix matrix_type;
	typedef DenseVector vector_type;

	const int m;
	const int n;
	const Real T;
	mutable DenseMatrix MA;
	mutable DenseMatrix MB;
	mutable DenseVector vg;

	ProblemBase(const int m, const int n, const double &T) : m(m), n(n), T(T), MA(m,n), MB(m,n), vg(m) 
	{
		MA.clear();
		MB.clear();
		vg.clear();
	}

	const DenseMatrix &A(const Real &) const { return MA; }
	const DenseMatrix &B(const Real &) const { return MB; }
	const DenseVector &g(const Real &) const { return vg; }

	const DenseMatrix &A(const Real &t, const DenseVector &u, const DenseVector &Du) const { return MA; }
	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const { return MB; }
};


/*
    This is a problem due to 

*/
struct ProblemDifficult : public ProblemBase
{
	typedef problem_linear problem_type;

	const Real eta;
	ProblemDifficult(const Real &eta = -0.2) : ProblemBase(2,2, 3.0), eta(eta) {}

	const DenseMatrix &A(const Real &t) const 
	{
		init(MA) = 0., 0., 1., eta*t;
		return MA;
	}

	const DenseMatrix &B(const Real &t) const 
	{
		init(MB) = 1., eta*t, 0., eta+1.;
		return MB;
	}

	const DenseVector &g(const Real &t) const
	{
		init(vg) = exp(-t), 0.;
		return vg;
	}
};	

struct ProblemNonunique : public ProblemBase
{
	typedef problem_linear problem_type;

	ProblemNonunique() : ProblemBase(2,2, 2.0) 
	{
		init(MB) = 1., 0.,
		           0., 1.;
	}

	const DenseMatrix &A(const Real &t) const 
	{
		init(MA) = -t, t*t, -1., t;
		return MA;
	}
};

// Example 2 from COLDAE paper of 1995.
struct ProblemBVP : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	enum IndexType
	{
		X1 = 0,
		X2, X3, Y
	};

	mutable DenseVector res;

	const Real epsilon;
	ProblemBVP(const Real &epsilon = 1.) : ProblemBase(4,4, 1.0), epsilon(epsilon) 
	{
		res.resize(4);

		MA(0, X1) = 1.;
		MA(1, X2) = 1.;
		MA(2, X3) = 1.;
	}

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
		using utility::sqr;

		init(res) = 
			Du[X1] - (epsilon + u[X2] - sin(t))*u[Y] - cos(t),
			Du[X2] - cos(t),
			Du[X3] - u[Y],
			(u[X1] - sin(t))*(u[Y] - exp(t));

		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
		MB(0, X2) = -u[Y];
		MB(0, Y) = -(epsilon + u[X2] - sin(t));
		MB(2, Y) = -1.;
		MB(3, X1) = u[Y] - exp(t);
		MB(3, Y) = u[X1] - sin(t);

		return MB;
	}
};

struct ProblemBVPSimplified : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	enum IndexType
	{
		X1 = 0,
		Y
	};

	mutable DenseVector res;

	const Real epsilon;
	ProblemBVPSimplified(const Real &epsilon = 1.) : ProblemBase(2,2, 1.0), epsilon(epsilon) 
	{
		res.resize(2);

		MA(0, X1) = 1.;
	}

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
		using utility::sqr;

		init(res) = 
			Du[X1] - (epsilon + sin(t) - sin(t))*u[Y] - cos(t),
			(u[X1] - sin(t))*(u[Y] - exp(t));

		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
		MB(0, Y) = -(epsilon + sin(t) - sin(t));
		MB(1, X1) = u[Y] - exp(t);
		MB(1, Y) = u[X1] - sin(t);

		return MB;
	}
};

struct ProblemSimple : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	enum IndexType
	{
		X1 = 0,
	};

	mutable DenseVector res;

	ProblemSimple() : ProblemBase(1,1, 1.0)
	{
		res.resize(1);

		MA(0, X1) = 1.;
	}

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
		using utility::sqr;

		init(res) = Du[X1];
//			Du[X1] - exp(t) - cos(t);

		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
		return MB;
	}
};

// Periodically driven electronic amplifier from Chapter 7 of MK06.
struct ProblemPDEA : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector res;

	ProblemPDEA() : ProblemBase(5,5, 0.01)
	{
		res.resize(5);

        const Real C1 = 1e-6;
        const Real C2 = 2e-6;
        const Real C3 = 3e-6;

        init(MA) =
            -C1, C1, 0., 0., 0.,
             C1,-C1, 0., 0., 0.,
             0., 0.,-C2, 0., 0.,
             0., 0., 0.,-C3, C3,
             0., 0., 0., C3,-C3;

	}

    const Real U_E(const Real &t) const { return 0.4 * sin(200.*M_PI*t); }
    const Real f(const Real &U) const { return 1e-6*(exp(U/0.026) - 1.); }
    const Real f_prime(const Real &U) const { return 1e-6/0.026*exp(U/0.026); }

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
        const Real U_B = 6.;
        const Real R0 = 1000.;
        const Real R1 = 9000.;
        const Real R2 = 9000.;
        const Real R3 = 9000.;
        const Real R4 = 9000.;
        const Real R5 = 9000.;
        const Real C1 = 1e-6;
        const Real C2 = 2e-6;
        const Real C3 = 3e-6;

		using utility::sqr;

		init(res) = 
            (U_E(t) - u[0])/R0 + C1*(Du[1] - Du[0]),
            (U_B - u[1])/R2 - u[1]/R1 + C1*(Du[0] - Du[1]) - 0.01*f(u[1]-u[2]),
            f(u[1]-u[2]) - u[2]/R3 - C2*Du[2],
            (U_B - u[3])/R4 + C3*(Du[4]-Du[3]) - 0.99*f(u[1]-u[2]),
            -u[4]/R5 + C3*(Du[3]-Du[4]);

		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
        const Real R0 = 1000.;
        const Real R1 = 9000.;
        const Real R2 = 9000.;
        const Real R3 = 9000.;
        const Real R4 = 9000.;
        const Real R5 = 9000.;

        MB(0,0) = -1./R0;
        
        MB(1,1) = -1./R2 - 1./R1 - 0.01*f_prime(u[1]-u[2])*1.;
        MB(1,2) = -0.01*f_prime(u[1]-u[2])*(-1.);

        MB(2,1) = f_prime(u[1]-u[2])*1.;
        MB(2,2) = f_prime(u[1]-u[2])*(-1.) - 1./R3;

        MB(3,1) = -0.99*f_prime(u[1]-u[2])*1.;
        MB(3,2) = -0.99*f_prime(u[1]-u[2])*(-1.);
        MB(3,3) = -1./R4;

        MB(4,4) = -1./R5;
		return MB;
	}
};

// Test Problem 13 from http://www.ma.ic.ac.uk/~jcash/BVP_software/readme.php#problems.
struct ProblemLinearOrdinaryBVP : public ProblemBase
{
	typedef problem_linear problem_type;

    const Real epsilon;
	ProblemLinearOrdinaryBVP(const Real &epsilon) : ProblemBase(2,2, 2.), epsilon(epsilon)
	{
        init(MA) =
            0., epsilon,
            1., 0.;

        init(MB) = 
            -1., 0.,
             0.,-1.;
	}

	const DenseVector &g(const Real &t) const
	{
		init(vg) = 
            -(epsilon*M_PI*M_PI + 1.)*cos(M_PI*(-1+t)),
            0.;
		return vg;
	}
};


struct ProblemNonlinearBVP : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	enum IndexType
	{
		X1 = 0,
		X2, Y, L1, L2, MU, NU, OM
	};

	mutable DenseVector res;

	const Real &omega;
	ProblemNonlinearBVP(const Real &omega) : ProblemBase(6, 8, 2.), omega(omega)
	{
		res.resize(m);

		MA(0, MU) = 1.;
		MA(1, L1) = 1.;
		MA(2, L2) = 1.;
		MA(3, NU) = 1.;
	}

	Real fun_r(const Real &t) const
	{
		return sin(omega*t)/omega + cos(omega*t);
	}
	
	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
		using utility::sqr;

		init(res) = 
			Du[OM], 
			Du[L1] + u[Y]*u[L1] - sqr(u[OM])*u[L2] + 2.*sqr(M_PI/3.)*u[X1]*u[MU] + (u[X1] + u[X2] - fun_r(t)),
			Du[L2] + u[L1] + u[Y]*u[L2] + 2.*u[X2]*u[MU] + (u[X1] + u[X2] - fun_r(t)), 
			Du[NU] - 2.*u[OM]*u[X1]*u[L2],
			sqr(M_PI/3.)*sqr(u[X1]) + sqr(u[X2]) - 1.,
			u[X1]*u[L1]+u[X2]*u[L2];
		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
		using utility::sqr;
		
		MB(1, Y) = u[L1];
		MB(1, L1) = u[Y];
		MB(1, OM) = -2.*u[OM]*u[L2];
		MB(1, L2) = -sqr(u[OM]);
		MB(1, X1) = 1. + 2.*sqr(M_PI/3.)*u[MU];
		MB(1, MU) = 2.*sqr(M_PI/3.)*u[X1];
		MB(1, X2) = 1.;

		MB(2, L1) = 1.;
		MB(2, Y) = u[L2];
		MB(2, L2) = u[Y];
		MB(2, X2) = 2.*u[MU] + 1.;
		MB(2, MU) = 2.*u[X2];
		MB(2, X1) = 1.;

		MB(3, OM) = -2.*u[X1]*u[L2];
		MB(3, X1) = -2.*u[OM]*u[L2];
		MB(3, L2) = -2.*u[OM]*u[X1];

		MB(4, X1) = 2.*sqr(M_PI/3.)*u[X1];
		MB(4, X2) = 2.*u[X2];

		MB(5, X1) = u[L1];
		MB(5, L1) = u[X1];
		MB(5, X2) = u[L2];
		MB(5, L2) = u[X2];
		return MB;
	}
};

// The following two problems are presented in Mahavier, A Numerical Method Utilizing Weighted Sobolev Descent To Solve Singular Differential Equations, 1997.
struct ProblemSingularity : public ProblemBase
{
	typedef problem_linear problem_type;

	ProblemSingularity() : ProblemBase(1,1, 1.0) 
	{
		init(MB) = -1.;
	}

	const DenseMatrix &A(const Real &t) const 
	{
		init(MA) = 2.*t;
		return MA;
	}
};	

struct ProblemSingularity2 : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector res;

	ProblemSingularity2() : ProblemBase(1,1, 1.)
	{
		assert(n == m);
		res.resize(n);
	}

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
		res(0) = t*t*Du(0) - 2.*t*u(0) - u(0)*u(0);
		return res;
	}
	const DenseMatrix &A(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
		MA(0,0) = t*t;
		return MA;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
		MB(0,0) = -2.*t - 2.*u(0);
		return MB;
	}
};

struct ProblemStiffness : public ProblemBase
{
	typedef problem_linear problem_type;

	ProblemStiffness() : ProblemBase(1,1, 1.0) 
	{
		const Real K = -15.;

		init(MA) = 1.;
		init(MB) = -1. * K;
	}
};	

struct ProblemConstantSmall : public ProblemBase
{
	typedef problem_linear problem_type;

	ProblemConstantSmall() : ProblemBase(2,2, 1.0)
	{
		init(MA) = 1., 1., 0., 0.;
		init(MB) = 0., 0., 0., 1.;
	}

	const DenseVector &g(const Real &t) const
	{
		init(vg) = exp(t)+cos(t), exp(t);
		return vg;
	}
};

struct ProblemConstantTest : public ProblemBase
{
	typedef problem_linear problem_type;

	ProblemConstantTest() : ProblemBase(2,2, 1.0)
	{
		init(MA) = 1., 0., 0., 0.;
		init(MB) = 0., 1., 0., 1.;
	}

	const DenseVector &g(const Real &t) const
	{
		init(vg) = 0, t;
		return vg;
	}
};

struct ProblemConstant : public ProblemBase
{
	typedef problem_linear problem_type;

	ProblemConstant(int m) : ProblemBase(m,m, 1.)
	{ 
		MA.clear();
		for (int i=0; i<m; i++)
		{
			if (i<m-1) MA(i,i+1) = 2;
			if (i>0) MA(i,i) = 1;
			if (i<m-2) MA(i,i+2) = 1;
		}
		MB.clear();
		for (int i=0; i<m; i++)
		{
			if (i<m-1) MB(i,i+1) = 1;
			MB(i,i) = 2;
			if (i>0) MB(i,i-1) = 1;
		}
	}

	const DenseVector &operator ()(const Real &t) const
	{
		vg(0) = -3.*t*t + 6.*t;
		for (int i=1; i<m-2; i++) vg(i) = -4.*t*t + 8.*t;
		vg(m-2) = -4.*t*t + 6.*t;
		vg(m-1) = -3.*t*t + 2.*t;
		return vg;
	}
};

struct ProblemSantiago2 : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	const Real Kprime;
	const Real M;

	DenseMatrix B1;
	mutable DenseVector res;

	ProblemSantiago2(const Real &Kprime = 1., const Real &M = 0.1) : ProblemBase(4,4, 5.), Kprime(Kprime), M(M) 
	{
		assert(n == m);
		B1.resize(n, n);
		res.resize(n);

		const Real KM = Kprime/M;
		init(B1) = 0., -1.,  0., 0.,
				   KM,  0., -KM, 0.,
				   0.,  0., -1., 0.,
				  -KM,  0.,  KM, 0.;

		MA = IdentityMatrix(n);
	}

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
		init(res) = 0., sin(u(0)), 0., sin(u(2));
		noalias(res) += Du;
		noalias(res) += prod(B1,u);
		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
		MB = B1;
		MB(1,0) += cos(u(0));
		MB(3,2) += cos(u(2));
		return MB;
	}

};

struct ProblemPendulumIndex3 : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector res;
    mutable DenseVector initial_y;

	ProblemPendulumIndex3() : ProblemBase(5,5, 1.)
	{
		assert(n == m);
		res.resize(n);
        initial_y.resize(n);

        init(MA) = 
            -1, 0, 0, 0, 0,
            0, -1, 0, 0, 0,
            0, 0, -1, 0, 0,
            0, 0, 0, -1, 0,
            0, 0, 0, 0, 0;
	}

	const DenseVector &F(const Real &t, const DenseVector &y, const DenseVector &yprime) const
	{
		DenseVector &df = res;

        df(0) = y(2);
        df(1) = y(3);
        df(2) = -y(0)*y(4);
        df(3) = -y(1)*y(4)-1.;
        df(4) = y(0)*y(0)+y(1)*y(1)-1.;

		ublas::noalias(res) += ublas::prod(MA, yprime);
		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &y, const DenseVector &yprime) const 
	{
        init(MB) =
            0, 0, 1, 0, 0,
            0, 0, 0, 1, 0,
            -y(4), 0, 0, 0, -y(0),
            0, -y(4), 0, 0, -y(1),
            2*y(0), 2*y(1), 0, 0, 0;
		return MB;
	}

	const DenseVector &initial() const
	{
        init(initial_y) = 1,0,0,1,1;
        return initial_y;
    }
};

struct ProblemPendulumIndex2 : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector res;
    mutable DenseVector initial_y;

	ProblemPendulumIndex2() : ProblemBase(5,5, 1.)
	{
		assert(n == m);
		res.resize(n);
        initial_y.resize(n);

        init(MA) = 
            -1, 0, 0, 0, 0,
            0, -1, 0, 0, 0,
            0, 0, -1, 0, 0,
            0, 0, 0, -1, 0,
            0, 0, 0, 0, 0;
	}

	const DenseVector &F(const Real &t, const DenseVector &y, const DenseVector &yprime) const
	{
		DenseVector &df = res;

        df(0)=y(2);
        df(1)=y(3);
        df(2)=-y(0)*y(4);
        df(3)=-y(1)*y(4)-1;
        df(4)=y(0)*y(2)+y(1)*y(3);

		ublas::noalias(res) += ublas::prod(MA, yprime);
		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &y, const DenseVector &yprime) const 
	{
        init(MB) =
            0, 0, 1, 0, 0,
            0, 0, 0, 1, 0,
            -y(4), 0, 0, 0, -y(0),
            0, -y(4), 0, 0, -y(1),
            y(2), y(3), y(0), y(1), 0;
		return MB;
	}

	const DenseVector &initial() const
	{
        init(initial_y) = 1,0,0,1,1;
        return initial_y;
    }
};

struct ProblemPendulumIndex1 : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector res;
    mutable DenseVector initial_y;

	ProblemPendulumIndex1() : ProblemBase(5,5, 1.)
	{
		assert(n == m);
		res.resize(n);
        initial_y.resize(n);

        init(MA) = 
            -1, 0, 0, 0, 0,
            0, -1, 0, 0, 0,
            0, 0, -1, 0, 0,
            0, 0, 0, -1, 0,
            0, 0, 0, 0, 0;
	}

	const DenseVector &F(const Real &t, const DenseVector &y, const DenseVector &yprime) const
	{
		DenseVector &df = res;

        df(0)=y(2);
        df(1)=y(3);
        df(2)=-y(0)*y(4);
        df(3)=-y(1)*y(4)-1;
        df(4)=y(2)*y(2)+y(3)*y(3)-y(4)-y(1);

		ublas::noalias(res) += ublas::prod(MA, yprime);
		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &y, const DenseVector &yprime) const 
	{
        init(MB) =
            0, 0, 1, 0, 0,
            0, 0, 0, 1, 0,
            -y(4), 0, 0, 0, -y(0),
            0, -y(4), 0, 0, -y(1),
            0, -1, 2*y(2), 2*y(3), -1;
		return MB;
	}

	const DenseVector &initial() const
	{
        init(initial_y) = 1,0,0,1,1;
        return initial_y;
    }
};

struct ProblemBenedikt : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	enum Style
	{
		WNT_NOSTIMULATION,
		WNT_STIMULATION,
		WNT_TRANSIENT,
		WNT_SINABS,
		WNT_COS
	};

	const Style style;

	Real K7;
	Real K8;
	Real K16;
	Real K17;
	Real k1;
	Real k2;
	Real k3;
	Real k4;
	Real k5;
	Real kp6;
	Real km6;
	Real k9;
	Real k10;
	Real k11;
	Real k13;
	Real k15;
	Real v12;
	Real v14;

	Real APC;
	Real TCF;
	Real GSK;
	Real Dsh;

	mutable DenseVector init_y;
	mutable DenseVector res;

	ProblemBenedikt(Style style) : ProblemBase(15, 15, 0.01), style(style)
	{
		init(MA) =
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, -1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, -1, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, -1, 0, 0, -1, 0, 0, -1, -1,
			0, 0, 0, 0, 0, -1, 0, 0, 0, 0, 0, -1, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0;

		init_y.resize(n);
		res.resize(m);

		// set problem parameters
		K7 = 100;
		K8 = 120;
		K16 = 30;
		K17 = 1200;
		k1 = 0.1818;
		k2 = 1.818e-2;
		k3 = 0.05;
		k4 = 0.2667;
		k5 = 0.1333;
		kp6 = 9.0191e-2;
		km6 = 0.909;
		k9 = 206;
		k10 = 206;
		k11 = 0.417;
		k13 = 2.489e-4;
		k15 = 0.167;
		v12 = 0.423;
		v14 = 8.2210e-5;
		APC = 100;
		TCF = 15;
		GSK = 50;
		Dsh = 100;	
	}

	Real W(const Real &t) const
	{
		switch (style)
		{
		case WNT_NOSTIMULATION:	
			return 0.;
		case WNT_STIMULATION:
			return 1.;
		case WNT_TRANSIENT:
			return exp(-t/20.);
		case WNT_SINABS:
			return fabs(sin(t));
		case WNT_COS:
			return (1.-cos(M_PI*t/750.))/2.;

		default:
			throw "ProblemBenedikt: this W(t) style was not implemented.";
		}
	}



	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
		const DenseVector &X = u;
		const DenseVector &XP = Du;

		init(res) = 
			-X[0]+Dsh-X[1],
			-XP[1]+k1*W(t)*X[0]-k2*X[1],
			-XP[2]-XP[7]+k4*X[3]-k5*X[2]-k9*X[7]+k10*X[8],
			-XP[3]-k3*X[1]*X[3]-k4*X[3]+k5*X[2]+kp6*X[4]*X[5]-km6*X[3],
			-X[4]+GSK,
			-X[5]+X[6]*X[11]/K7,
			-X[6]+K17*APC/(K17+X[10]),
			-X[7]+X[2]*X[10]/K8,
			-XP[8]+k9*X[7]-k10*X[8],
			-XP[9]+k10*X[8]-k11*X[9],
			-XP[10]-XP[7]-XP[13]-XP[14]-k9*X[7]+v12-k13*X[10],
			-XP[11]-XP[5]+k3*X[1]*X[3]-kp6*X[4]*X[5]+km6*X[3]+v14-k15*X[11],
			-X[12]+K16*TCF/(K16+X[10]),
			-X[13]+X[10]*TCF/(K16+X[10]),
			-X[14]+X[10]*APC/(K17+X[10]);

		return res;
	}


	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
		using utility::sqr;
		const DenseVector &X = u;

		init(MB) = 
			-1, -1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			k1*W(t), -k2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, -k5, k4, 0, 0, 0, -k9, k10, 0, 0, 0, 0, 0, 0,
			0, -k3*X[3], k5, -k3*X[1]-k4-km6, kp6*X[5], kp6*X[4], 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, -1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, -1, X[11]/K7, 0, 0, 0, 0, X[6]/K7, 0, 0, 0,
			0, 0, 0, 0, 0, 0, -1, 0, 0, 0, -K17*APC/sqr(K17+X[10]), 0, 0, 0, 0,
			0, 0, X[10]/K8, 0, 0, 0, 0, -1, 0, 0, X[2]/K8, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, k9, -k10, 0, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, k10, -k11, 0, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, -k9, 0, 0, -k13, 0, 0, 0, 0,
			0, k3*X[3], 0, k3*X[1]+km6, -kp6*X[5], -kp6*X[4], 0, 0, 0, 0, 0, -k15, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -K16*TCF/sqr(K16+X[10]), 0, -1, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, TCF/(K16+X[10])-X[10]*TCF/sqr(K16+X[10]), 0, 0, -1, 0,
			0, 0, 0, 0, 0, 0, 0, 0, 0, 0, APC/(K17+X[10])-X[10]*APC/sqr(K17+X[10]), 0, 0, 0, -1;
		return MB;
	}



	const DenseVector &initial() const
	{
		init(init_y) =
			100.,
			0.,
			9.66e-3,
			4.83e-3,
			50.,
			9.66e-4,
			97.95,
			2.02e-3,
			2.02e-3,
			1.,
			25.11,
			4.93e-4,
			8.17,
			6.83,
			2.05;

		return init_y;

	}



};

struct ProblemTrigonometrical : public ProblemBase
{
	typedef problem_nonlinear problem_type;  // yes, this is meant this way

	mutable DenseVector res;

	ProblemTrigonometrical() : ProblemBase(2,2, 2.*M_PI)
	{
		res.resize(n);

		MA = IdentityMatrix(n);
		init(MB) =
			0.0, -1.0,
			1.0,  0.0;
	}

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
		noalias(res) = prod(MA, Du) + prod(MB, u);
		return res;
	}
};

struct ProblemSimpleNonLinear : public ProblemBase
{
	// Problem:   y' - exp(-y) = 0
	typedef problem_nonlinear problem_type;

	mutable DenseVector res;

	ProblemSimpleNonLinear() : ProblemBase(1,1, 10.)
	{
		res.resize(n);

		MA(0,0) = 1.;
		vg(0) = 0.;
	}

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
		res(0) = Du(0) - exp(-u(0));
		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
		MB(0,0) = exp(-u(0));
		return MB;
	}

};

struct ProblemScaling : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector res;

	ProblemScaling() : ProblemBase(2,2, 2.)
	{
		res.resize(n);

		MA(0,0) = 1.;
	}

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
		init(res) = Du(0) - cos(t),
            u(1) - exp(7.*u(0));
		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
        MB(1,0) = -7.*exp(7.*u(0));
        MB(1,1) = 1.;
		return MB;
	}

};



struct ProblemChemakzo : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector f;
	mutable DenseVector sol;
	mutable DenseVector init_y;
	mutable DenseVector init_yprime;

	ProblemChemakzo(const Real &T = 180.) : ProblemBase(6,6, T)
	{
		f.resize(n);
		sol.resize(n);
		init_y.resize(n);
		init_yprime.resize(n);

		init(MA) = 
			-1, 0, 0, 0, 0, 0,
			 0,-1, 0, 0, 0, 0,
			 0, 0,-1, 0, 0, 0,
			 0, 0, 0,-1, 0, 0,
			 0, 0, 0, 0,-1, 0,
			 0, 0, 0, 0, 0, 0;
	}

	const DenseVector &F(const Real &t, const DenseVector &y, const DenseVector &yprime) const
	{
		const double k1   = 18.7;
		const double k2   = 0.58;
		const double k3   = 0.09;
		const double k4   = 0.42;
		const double kbig = 34.4;
		const double kla  = 3.3;
		const double ks   = 115.83;
		const double po2  = 0.9;
		const double hen  = 737;

//		assert(y(1) > 0.);
//		if (y(1) <= 0.) throw InvalidInput("Problem chemakzo: y(1)<0 and we need square root.");


		double r1  = k1*pow(y(0),4)*sqrt(fabs(y(1)));
		double r2  = k2*y(2)*y(3);
		double r3  = k2/kbig*y(0)*y(4);
		double r4  = k3*y(0)*pow(y(3),2);
		double r5  = k4*pow(y(5),2)*sqrt(fabs(y(1)));
		double fin = kla*(po2/hen-fabs(y(1)));

		f(0) =   -2*r1 +r2 -r3     -r4;
		f(1) = -0.5*r1             -r4     -0.5*r5 + fin;
		f(2) =        r1 -r2 +r3;
		f(3) =           -r2 +r3 -2*r4;
		f(4) =            r2 -r3         +r5;
		f(5) = ks*y(0)*y(3)-y(5);

		ublas::noalias(f) += ublas::prod(MA, yprime);
		return f;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &y, const DenseVector &yprime) const 
	{
		using utility::sign;
		DenseMatrix &dfdy = MB;

		const double k1   = 18.7;
		const double k2   = 0.58;
		const double k3   = 0.09;
		const double k4   = 0.42;
		const double kbig = 34.4;
		const double kla  = 3.3;
		const double ks   = 115.83;

//	  	assert(y(1) > 0.);
//		if (y(1) <= 0.) throw InvalidInput("Problem chemakzo: y(1)<0 and we need square root.");

		const double r11  = 4*k1*pow(y(0),3)*sqrt(fabs(y(1)))*sign(y(1));
		const double r12  = 0.5*k1*pow(y(0),4)/sqrt(fabs(y(1)))*sign(y(1));
		const double r23  = k2*y(3);
		const double r24  = k2*y(2);
		const double r31  = (k2/kbig)*y(4);
		const double r35  = (k2/kbig)*y(0);
		const double r41  = k3*pow(y(3),2);
		const double r44  = 2*k3*y(0)*y(3);
		const double r52  = 0.5*k4*pow(y(5),2)/sqrt(fabs(y(1)))*sign(y(1));
		const double r56  = 2*k4*y(5)*sqrt(fabs(y(1)))*sign(y(1));
		const double fin2 = -kla;

		dfdy(0,0) = -2*r11-r31-r41;
		dfdy(0,1) = -2*r12;
		dfdy(0,2) = r23;
		dfdy(0,3) = r24-r44;
		dfdy(0,4) = -r35;
		dfdy(1,0) = -0.5*r11-r41;
		dfdy(1,1) = -0.5*r12-0.5*r52+fin2;
		dfdy(1,3) = -r44;
		dfdy(1,5) = -0.5*r56;
		dfdy(2,0) = r11+r31;
		dfdy(2,1) = r12;
		dfdy(2,2) = -r23;
		dfdy(2,3) = -r24;
		dfdy(2,4) = r35;
		dfdy(3,0) = r31-2*r41;
		dfdy(3,2) = -r23;
		dfdy(3,3) = -r24-2*r44;
		dfdy(3,4) = r35;
		dfdy(4,0) = -r31;
		dfdy(4,1) = r52;
		dfdy(4,2) = r23;
		dfdy(4,3) = r24;
		dfdy(4,4) = -r35;
		dfdy(4,5) = r56;
		dfdy(5,0) = ks*y(3);
		dfdy(5,3) = ks*y(0);
		dfdy(5,5) = -1;

		return MB;
	}

	const DenseVector &initial() const
	{
		DenseVector &y = init_y;

     	const double ks   = 115.83;
		y(0) = 0.444;
		y(1) = 0.00123;
		y(2) = 0;
		y(3) = 0.007;
		y(4) = 0;
		y(5) = ks*y(0)*y(3);

		return init_y;
	}

	const DenseVector &initial_deriv() const
	{
		init_yprime = F(0, initial(), ScalarVector(n, 0.));
		return init_yprime;
	}

	const DenseVector &solution() const
	{
		DenseVector &y = sol;

		y(0) = 0.1150794920661702e0;
		y(1) = 0.1203831471567715e-2;
		y(2) = 0.1611562887407974e0;
		y(3) = 0.3656156421249283e-3;
		y(4) = 0.1708010885264404e-1;
		y(5) = 0.4873531310307455e-2;

		return sol;
	}

};


// Problem 13 of the initial value problemset: Andrew's squeezing mechanism


struct ProblemAndrewsConstants
{
    const Real m1;
    const Real m2;
    const Real m3;
    const Real m4;
    const Real m5;
    const Real m6;
    const Real m7;
    const Real xa;
    const Real ya;
    const Real xb;
    const Real yb;
    const Real xc;
    const Real yc;
    const Real c0;
    const Real i1;
    const Real i2;
    const Real i3;
    const Real i4;
    const Real i5;
    const Real i6;
    const Real i7;
    const Real d;
    const Real da;
    const Real e;
    const Real ea;
    const Real rr;
    const Real ra;
    const Real l0;
    const Real ss;
    const Real sa;
    const Real sb;
    const Real sc;
    const Real sd;
    const Real ta;
    const Real tb;
    const Real u;
    const Real ua;
    const Real ub;
    const Real zf;
    const Real zt;
    const Real fa;
    const Real mom;

    ProblemAndrewsConstants() :
        m1(.04325e0),
        m2(.00365e0),
        m3(.02373e0),
        m4(.00706e0),
        m5(.07050e0),
        m6(.00706e0),
        m7(.05498e0),
        xa(-.06934e0),
        ya(-.00227e0),
        xb(-0.03635e0),
        yb(.03273e0),
        xc(.014e0),
        yc(.072e0),
        c0(4530e0),
        i1(2.194e-6),
        i2(4.410e-7),
        i3(5.255e-6),
        i4(5.667e-7),
        i5(1.169e-5),
        i6(5.667e-7),
        i7(1.912e-5),
        d(28e-3),
        da(115e-4),
        e(2e-2),
        ea(1421e-5),
        rr(7e-3),
        ra(92e-5),
        l0(7785e-5),
        ss(35e-3),
        sa(1874e-5),
        sb(1043e-5),
        sc(18e-3),
        sd(2e-2),
        ta(2308e-5),
        tb(916e-5),
        u(4e-2),
        ua(1228e-5),
        ub(449e-5),
        zf(2e-2),
        zt(4e-2),
        fa(1421e-5),
        mom(33e-3)
    {}
};

struct ProblemAndrewsIndex3 : public ProblemBase, public ProblemAndrewsConstants
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector ff;
	mutable DenseVector init_y;
	mutable DenseVector final_y;
	mutable DenseVector init_yprime;

	ProblemAndrewsIndex3(const Real &T = 3e-2) : ProblemBase(27,27, T)
	{
		ff.resize(m);
		init_y.resize(n);
		final_y.resize(n);
		init_yprime.resize(n);

		for (int i=0; i<14; i++)
			MA(i,i) = -1.0;
	}

	const DenseVector &F(const Real &t, const DenseVector &y, const DenseVector &yprime) const
	{
		using utility::sqr;

		const Real sibe = sin(y(0));
		const Real sith = sin(y(1));
		const Real siga = sin(y(2));
		const Real siph = sin(y(3));
		const Real side = sin(y(4));
		const Real siom = sin(y(5));
		const Real siep = sin(y(6));

		const Real cobe = cos(y(0));
		const Real coth = cos(y(1));
		const Real coga = cos(y(2));
		const Real coph = cos(y(3));
		const Real code = cos(y(4));
		const Real coom = cos(y(5));
		const Real coep = cos(y(6));

		const Real sibeth = sin(y(0)+y(1));
		const Real siphde = sin(y(3)+y(4));
		const Real siomep = sin(y(5)+y(6));

		const Real cobeth = cos(y(0)+y(1));
		const Real cophde = cos(y(3)+y(4));
		const Real coomep = cos(y(5)+y(6));

		const Real bep = y(7);
		const Real thp = y(8);
		const Real php = y(10);
		const Real dep = y(11);
		const Real omp = y(12);
		const Real epp = y(13);

		DenseMatrix m(7,7);
		m.clear();
		m(0,0) = m1*sqr(ra) + m2*(sqr(rr)-2*da*rr*coth+sqr(da)) + i1 + i2;
		m(1,0) = m2*(sqr(da)-da*rr*coth) + i2;
		m(1,1) = m2*sqr(da) + i2;
		m(2,2) = m3*(sqr(sa)+sqr(sb)) + i3;
		m(3,3) = m4*sqr(e-ea) + i4;
		m(4,3) = m4*(sqr(e-ea)+zt*(e-ea)*siph) + i4;
		m(4,4) = m4*(sqr(zt)+2*zt*(e-ea)*siph+sqr(e-ea)) + m5*(sqr(ta)+sqr(tb)) + i4 + i5;
		m(5,5) = m6*sqr(zf-fa) + i6;
		m(6,5) = m6*(sqr(zf-fa)-u*(zf-fa)*siom) + i6;
		m(6,6) = m6*(sqr(zf-fa)-2*u*(zf-fa)*siom+sqr(u)) + m7*(sqr(ua)+sqr(ub)) + i6 + i7;

		for (int j=1; j<7; j++)
			for (int i=0; i<j; i++)
				m(i,j) = m(j,i);

		const Real xd = sd*coga + sc*siga + xb;
		const Real yd = sd*siga - sc*coga + yb;
		const Real lang  = sqrt(sqr(xd-xc) + sqr(yd-yc));
		const Real force = - c0 * (lang - l0)/lang;
		const Real fx = force * (xd-xc);
		const Real fy = force * (yd-yc);
	
		DenseVector f(7);
		f(0) = mom - m2*da*rr*thp*(thp+2*bep)*sith;
		f(1) = m2*da*rr*sqr(bep)*sith;
		f(2) = fx*(sc*coga - sd*siga) + fy*(sd*coga + sc*siga);
		f(3) = m4*zt*(e-ea)*sqr(dep)*coph;
		f(4) = - m4*zt*(e-ea)*php*(php+2*dep)*coph;
		f(5) = - m6*u*(zf-fa)*sqr(epp)*coom;
		f(6) = m6*u*(zf-fa)*omp*(omp+2*epp)*coom;


		DenseMatrix gp(6,7);
		gp.clear();
		gp(0,0) = - rr*sibe + d*sibeth;
		gp(0,1) = d*sibeth;
		gp(0,2) = - ss*coga;
		gp(1,0) = rr*cobe - d*cobeth;
		gp(1,1) = - d*cobeth;
		gp(1,2) = - ss*siga;
		gp(2,0) = - rr*sibe + d*sibeth;
		gp(2,1) = d*sibeth;
		gp(2,3) = - e*cophde;
		gp(2,4) = - e*cophde + zt*side;
		gp(3,0) = rr*cobe - d*cobeth;
		gp(3,1) = - d*cobeth;
		gp(3,3) = - e*siphde;
		gp(3,4) = - e*siphde - zt*code;
		gp(4,0) = - rr*sibe + d*sibeth;
		gp(4,1) = d*sibeth;
		gp(4,5) = zf*siomep;
		gp(4,6) = zf*siomep - u*coep;
		gp(5,0) = rr*cobe - d*cobeth;
		gp(5,1) = - d*cobeth;
		gp(5,5) = - zf*coomep;
		gp(5,6) = - zf*coomep - u*siep;

		DenseVector g(6);
		g(0) = rr*cobe - d*cobeth - ss*siga - xb;
		g(1) = rr*sibe - d*sibeth + ss*coga - yb;
		g(2) = rr*cobe - d*cobeth - e*siphde - zt*code - xa;
		g(3) = rr*sibe - d*sibeth + e*cophde - zt*side - ya;
		g(4) = rr*cobe - d*cobeth - zf*coomep - u*siep - xa;
		g(5) = rr*sibe - d*sibeth - zf*siomep + u*coep - ya;

		for (int i=0; i<14; i++)
			ff(i) = y(i+7);

		for (int i=14; i<21; i++)
		{
			ff(i) = -f(i-14);
			for (int j=0; j<7; j++)
				ff(i) += m(i-14,j)*y(j+14);
			for (int j=0; j<6; j++)
				ff(i) += gp(j,i-14)*y(j+21);
		}
		for (int i=21; i<27; i++)
			ff(i) = g(i-21);

		// that is the only difference to the FORTRAN interface...	
		ublas::noalias(ff) += ublas::prod(MA, yprime);
		return ff;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &y, const DenseVector &yprime) const 
	{
		using utility::sqr;
		DenseMatrix &dfdy = MB;

		//-----------------------------------------------------------------------
		//     the Jacobian computed here is an approximation, see p. 540 of
		//     Hairer & Wanner 'solving ordinary differential equations II'
		//-----------------------------------------------------------------------

		const Real sibe = sin(y(0));
		const Real siga = sin(y(2));
		const Real siph = sin(y(3));
		const Real side = sin(y(4));
		const Real siom = sin(y(5));
		const Real siep = sin(y(6));

		const Real cobe = cos(y(0));
		const Real coth = cos(y(1));
		const Real coga = cos(y(2));
		const Real code = cos(y(4));
		const Real coep = cos(y(6));

		const Real sibeth = sin(y(0)+y(1));
		const Real siphde = sin(y(3)+y(4));
		const Real siomep = sin(y(5)+y(6));

		const Real cobeth = cos(y(0)+y(1));
		const Real cophde = cos(y(3)+y(4));
		const Real coomep = cos(y(5)+y(6));


		DenseMatrix m(7,7);
		m.clear();
		m(0,0) = m1*sqr(ra) + m2*(sqr(rr)-2*da*rr*coth+sqr(da)) + i1 + i2;
		m(1,0) = m2*(sqr(da)-da*rr*coth) + i2;
		m(1,1) = m2*sqr(da) + i2;
		m(2,2) = m3*(sqr(sa)+sqr(sb)) + i3;
		m(3,3) = m4*sqr(e-ea) + i4;
		m(4,3) = m4*(sqr(e-ea)+zt*(e-ea)*siph) + i4;
		m(4,4) = m4*(sqr(zt)+2*zt*(e-ea)*siph+sqr(e-ea)) + m5*(sqr(ta)+sqr(tb)) + i4 + i5;
		m(5,5) = m6*sqr(zf-fa) + i6;
		m(6,5) = m6*(sqr(zf-fa)-u*(zf-fa)*siom) + i6;
		m(6,6) = m6*(sqr(zf-fa)-2*u*(zf-fa)*siom+sqr(u)) + m7*(sqr(ua)+sqr(ub)) + i6 + i7;

		for (int j=1; j<7; j++)
			for (int i=0; i<j; i++)
				m(i,j) = m(j,i);

		DenseMatrix gp(6,7);
		gp.clear();
		gp(0,0) = - rr*sibe + d*sibeth;
		gp(0,1) = d*sibeth;
		gp(0,2) = - ss*coga;
		gp(1,0) = rr*cobe - d*cobeth;
		gp(1,1) = - d*cobeth;
		gp(1,2) = - ss*siga;
		gp(2,0) = - rr*sibe + d*sibeth;
		gp(2,1) = d*sibeth;
		gp(2,3) = - e*cophde;
		gp(2,4) = - e*cophde + zt*side;
		gp(3,0) = rr*cobe - d*cobeth;
		gp(3,1) = - d*cobeth;
		gp(3,3) = - e*siphde;
		gp(3,4) = - e*siphde - zt*code;
		gp(4,0) = - rr*sibe + d*sibeth;
		gp(4,1) = d*sibeth;
		gp(4,5) = zf*siomep;
		gp(4,6) = zf*siomep - u*coep;
		gp(5,0) = rr*cobe - d*cobeth;
		gp(5,1) = - d*cobeth;
		gp(5,5) = - zf*coomep;
		gp(5,6) = - zf*coomep - u*siep;


		for (int i=0; i<14; i++)
			dfdy(i,i+7) = 1e0;

		for (int i=0; i<7; i++)
			for (int j=0; j<7; j++)
				dfdy(14+j,14+i) = m(j,i);
				
		for (int i=0; i<6; i++)
			for (int j=0; j<7; j++)
				dfdy(14+j,21+i) = gp(i,j);

		for (int i=0; i<7; i++)
			for (int j=0; j<6; j++)
				dfdy(21+j,i) = gp(j,i);
        
        if (0)
        { // this particularly ugly code was generated using Maple 

            // These are the missing terms that have been left out in the approximation from Hairer & Wanner.
            // Not surprisingly, they don't change much...
            Real Dq[7][7];
            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                    Dq[i][j] = 0.;
            Dq[0][0] = (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[21] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[22] + (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[23] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[24] + (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[25] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[26];
            Dq[0][1] = 0.2e1 * m2 * da * rr * sin(y[1]) * y[14] + m2 * da * rr * sin(y[1]) * y[15] + m2 * da * rr * y[8] * (y[8] + 0.2e1 * y[7]) * cos(y[1]) + d * cos(y[0] + y[1]) * y[21] + d * sin(y[0] + y[1]) * y[22] + d * cos(y[0] + y[1]) * y[23] + d * sin(y[0] + y[1]) * y[24] + d * cos(y[0] + y[1]) * y[25] + d * sin(y[0] + y[1]) * y[26];
            Dq[1][0] = d * cos(y[0] + y[1]) * y[21] + d * sin(y[0] + y[1]) * y[22] + d * cos(y[0] + y[1]) * y[23] + d * sin(y[0] + y[1]) * y[24] + d * cos(y[0] + y[1]) * y[25] + d * sin(y[0] + y[1]) * y[26];
            Dq[1][1] = m2 * da * rr * sin(y[1]) * y[14] - m2 * da * rr * pow(y[7], 0.2e1) * cos(y[1]) + d * cos(y[0] + y[1]) * y[21] + d * sin(y[0] + y[1]) * y[22] + d * cos(y[0] + y[1]) * y[23] + d * sin(y[0] + y[1]) * y[24] + d * cos(y[0] + y[1]) * y[25] + d * sin(y[0] + y[1]) * y[26];
            Dq[2][2] = c0 / (pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) 
                * (sc * cos(y[2]) - sd * sin(y[2])) / 0.2e1 - c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.3e1 / 0.2e1) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) / 0.2e1 + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * pow(sc * cos(y[2]) - sd * sin(y[2]), 0.2e1) + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (-sc * sin(y[2]) - sd * cos(y[2])) + c0 / (pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2])) / 0.2e1 - c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.3e1 / 0.2e1) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2])) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) / 0.2e1 + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * pow(sd * cos(y[2]) + sc * sin(y[2]), 0.2e1) + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sc * cos(y[2]) - sd * sin(y[2])) + ss * sin(y[2]) * y[21] - ss * cos(y[2]) * y[22];
            Dq[3][3] = m4 * zt * (e - ea) * cos(y[3]) * y[18] + m4 * zt * (e - ea) * pow(y[11], 0.2e1) * sin(y[3]) + e * sin(y[3] + y[4]) * y[23] - e * cos(y[3] + y[4]) * y[24];
            Dq[3][4] = e * sin(y[3] + y[4]) * y[23] - e * cos(y[3] + y[4]) * y[24];
            Dq[4][3] = m4 * zt * (e - ea) * cos(y[3]) * y[17] + 0.2e1 * m4 * zt * (e - ea) * cos(y[3]) * y[18] - m4 * zt * (e - ea) * y[10] * (y[10] + 0.2e1 * y[11]) * sin(y[3]) + e * sin(y[3] + y[4]) * y[23] - e * cos(y[3] + y[4]) * y[24];
            Dq[4][4] = (e * sin(y[3] + y[4]) + zt * cos(y[4])) * y[23] + (-e * cos(y[3] + y[4]) + zt * sin(y[4])) * y[24];
            Dq[5][5] = -m6 * u * (zf - fa) * cos(y[5]) * y[20] - m6 * u * (zf - fa) * pow(y[13], 0.2e1) * sin(y[5]) + zf * cos(y[5] + y[6]) * y[25] + zf * sin(y[5] + y[6]) * y[26];
            Dq[5][6] = zf * cos(y[5] + y[6]) * y[25] + zf * sin(y[5] + y[6]) * y[26];
            Dq[6][5] = -m6 * u * (zf - fa) * cos(y[5]) * y[19] - 0.2e1 * m6 * u * (zf - fa) * cos(y[5]) * y[20] + m6 * u * (zf - fa) * y[12] * (y[12] + 0.2e1 * y[13]) * sin(y[5]) + zf * cos(y[5] + y[6]) * y[25] + zf * sin(y[5] + y[6]) * y[26];
            Dq[6][6] = (zf * cos(y[5] + y[6]) + u * sin(y[6])) * y[25] + (zf * sin(y[5] + y[6]) - u * cos(y[6])) * y[26];

            
            Real Dqp[7][7];
            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                    Dqp[i][j] = 0.;
            Dqp[0][0] = 0.2e1 * m2 * da * rr * y[8] * sin(y[1]);
            Dqp[0][1] = m2 * da * rr * (y[8] + 0.2e1 * y[7]) * sin(y[1]) + m2 * da * rr * y[8] * sin(y[1]);
            Dqp[1][0] = -0.2e1 * m2 * da * rr * y[7] * sin(y[1]);
            Dqp[3][4] = -0.2e1 * m4 * zt * (e - ea) * y[11] * cos(y[3]);
            Dqp[4][3] = m4 * zt * (e - ea) * (y[10] + 0.2e1 * y[11]) * cos(y[3]) + m4 * zt * (e - ea) * y[10] * cos(y[3]);
            Dqp[4][4] = 0.2e1 * m4 * zt * (e - ea) * y[10] * cos(y[3]);
            Dqp[5][6] = 0.2e1 * m6 * u * (zf - fa) * y[13] * cos(y[5]);
            Dqp[6][5] = -m6 * u * (zf - fa) * (y[12] + 0.2e1 * y[13]) * cos(y[5]) - m6 * u * (zf - fa) * y[12] * cos(y[5]);
            Dqp[6][6] = -0.2e1 * m6 * u * (zf - fa) * y[12] * cos(y[5]);

            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                {
                    dfdy(i+14,j) = Dq[i][j];
                    dfdy(i+14,j+7) = Dqp[i][j];
                }
        }

		return MB;
	}


	const DenseVector &initial() const
	{
		DenseVector &y = init_y;

		y(0)  = -0.0617138900142764496358948458001e0;
		y(1)  =  0e0;
		y(2)  =  0.455279819163070380255912382449e0;
		y(3)  =  0.222668390165885884674473185609e0;
		y(4)  =  0.487364979543842550225598953530e0;
		y(5)  = -0.222668390165885884674473185609e0;
		y(6)  =  1.23054744454982119249735015568e0;
		y(7)  =  0e0;
		y(8)  =  0e0;
		y(9)  =  0e0;
		y(10) =  0e0;
		y(11) =  0e0;
		y(12) =  0e0;
		y(13) =  0e0;
		y(14) =  14222.4439199541138705911625887e0;
		y(15) = -10666.8329399655854029433719415e0;
		y(16) =  0e0;
		y(17) =  0e0;
		y(18) =  0e0;
		y(19) =  0e0;
		y(20) =  0e0;
		y(21) =  98.5668703962410896057654982170e0;
		y(23) = -6.12268834425566265503114393122e0;
		y(23) =  0e0;
		y(24) =  0e0;
		y(25) =  0e0;
		y(26) =  0e0;

		return init_y;
	}

	const DenseVector &final() const
	{
		DenseVector &y = final_y;

		y(0)  = 15.8107711941657794;
		y(1)  = -15.7563710571773807;
		y(2)  = 0.0408222401100961;
		y(3)  = -0.5347301163587794;
		y(4)  = 0.5244099658794575;
		y(5)  = 0.5347301163587794;
		y(6)  = 1.0480807410416579;
		y(7)  = 1139.9203007108333168;
		y(8)  = -1424.3792932389776524;
		y(9)  = 11.0329116148355091;
		y(10) = 19.2933735871602501;
		y(11) = 0.5735698974635138;
		y(12) = -19.2933735869326171;
		y(13) = 0.3231791462295504;
		y(14) = -24631.7008203593468352;
		y(15) = 51850.2920066463047988;
		y(16) = 324102.5832561646238901;
		y(17) = 566749.3904016350861639;
		y(18) = 16743.6304924886462686;
		y(19) = -566749.3894865390611812;
		y(20) = 9826.5193467981262074;
		y(21) = 199.1753399486409251;
		y(23) = -29.7553083114498733;
		y(23) = 23.0665423222055033;
		y(24) = 31.4527159258740880;
		y(25) = 22.6424934101455264;
		y(26) = 11.6174053760077776;

		return final_y;
	}

	const DenseVector &initial_deriv() const
	{
		initial();
		for (int i=0; i<14; i++)
			init_yprime(i) = init_y(i+7);
		for (int i=14; i<27; i++)
			init_yprime(i) = 0e0;
		return init_yprime;
	}
};


struct ProblemAndrewsIndex2Small : public ProblemBase, public ProblemAndrewsConstants
{
	typedef problem_nonlinear problem_type;

    mutable DenseVector phi;
	mutable DenseVector ff;
	mutable DenseVector init_y;
	mutable DenseVector final_y;

	ProblemAndrewsIndex2Small(const Real &T=3e-2) : ProblemBase(20,20, T)
	{
        phi.resize(m);
		ff.resize(m);
		init_y.resize(n);
		final_y.resize(n);
	}

    const DenseVector &Phi(const Real &t, const DenseVector &y) const
    {
		using utility::sqr;

		const Real sibe = sin(y(0));
		const Real sith = sin(y(1));
		const Real siga = sin(y(2));
		const Real siph = sin(y(3));
		const Real side = sin(y(4));
		const Real siom = sin(y(5));
		const Real siep = sin(y(6));

		const Real cobe = cos(y(0));
		const Real coth = cos(y(1));
		const Real coga = cos(y(2));
		const Real coph = cos(y(3));
		const Real code = cos(y(4));
		const Real coom = cos(y(5));
		const Real coep = cos(y(6));

		const Real sibeth = sin(y(0)+y(1));
		const Real siphde = sin(y(3)+y(4));
		const Real siomep = sin(y(5)+y(6));

		const Real cobeth = cos(y(0)+y(1));
		const Real cophde = cos(y(3)+y(4));
		const Real coomep = cos(y(5)+y(6));

		const Real bep = y(7);
		const Real thp = y(8);
		const Real php = y(10);
		const Real dep = y(11);
		const Real omp = y(12);
		const Real epp = y(13);

		DenseMatrix m(7,7);
		m.clear();
		m(0,0) = m1*sqr(ra) + m2*(sqr(rr)-2*da*rr*coth+sqr(da)) + i1 + i2;
		m(1,0) = m2*(sqr(da)-da*rr*coth) + i2;
		m(1,1) = m2*sqr(da) + i2;
		m(2,2) = m3*(sqr(sa)+sqr(sb)) + i3;
		m(3,3) = m4*sqr(e-ea) + i4;
		m(4,3) = m4*(sqr(e-ea)+zt*(e-ea)*siph) + i4;
		m(4,4) = m4*(sqr(zt)+2*zt*(e-ea)*siph+sqr(e-ea)) + m5*(sqr(ta)+sqr(tb)) + i4 + i5;
		m(5,5) = m6*sqr(zf-fa) + i6;
		m(6,5) = m6*(sqr(zf-fa)-u*(zf-fa)*siom) + i6;
		m(6,6) = m6*(sqr(zf-fa)-2*u*(zf-fa)*siom+sqr(u)) + m7*(sqr(ua)+sqr(ub)) + i6 + i7;

		for (int j=1; j<7; j++)
			for (int i=0; i<j; i++)
				m(i,j) = m(j,i);

		const Real xd = sd*coga + sc*siga + xb;
		const Real yd = sd*siga - sc*coga + yb;
		const Real lang  = sqrt(sqr(xd-xc) + sqr(yd-yc));
		const Real force = - c0 * (lang - l0)/lang;
		const Real fx = force * (xd-xc);
		const Real fy = force * (yd-yc);
	
		DenseVector f(7);
		f(0) = mom - m2*da*rr*thp*(thp+2*bep)*sith;
		f(1) = m2*da*rr*sqr(bep)*sith;
		f(2) = fx*(sc*coga - sd*siga) + fy*(sd*coga + sc*siga);
		f(3) = m4*zt*(e-ea)*sqr(dep)*coph;
		f(4) = - m4*zt*(e-ea)*php*(php+2*dep)*coph;
		f(5) = - m6*u*(zf-fa)*sqr(epp)*coom;
		f(6) = m6*u*(zf-fa)*omp*(omp+2*epp)*coom;


		DenseMatrix gp(6,7);
		gp.clear();
		gp(0,0) = - rr*sibe + d*sibeth;
		gp(0,1) = d*sibeth;
		gp(0,2) = - ss*coga;
		gp(1,0) = rr*cobe - d*cobeth;
		gp(1,1) = - d*cobeth;
		gp(1,2) = - ss*siga;
		gp(2,0) = - rr*sibe + d*sibeth;
		gp(2,1) = d*sibeth;
		gp(2,3) = - e*cophde;
		gp(2,4) = - e*cophde + zt*side;
		gp(3,0) = rr*cobe - d*cobeth;
		gp(3,1) = - d*cobeth;
		gp(3,3) = - e*siphde;
		gp(3,4) = - e*siphde - zt*code;
		gp(4,0) = - rr*sibe + d*sibeth;
		gp(4,1) = d*sibeth;
		gp(4,5) = zf*siomep;
		gp(4,6) = zf*siomep - u*coep;
		gp(5,0) = rr*cobe - d*cobeth;
		gp(5,1) = - d*cobeth;
		gp(5,5) = - zf*coomep;
		gp(5,6) = - zf*coomep - u*siep;

		for (int i=0; i<7; i++)
			phi(i) = y(i+7);

		for (int i=7; i<14; i++)
		{
			phi(i) = -f(i-7);
			for (int j=0; j<6; j++)
				phi(i) += gp(j,i-7)*y(j+14);
		}
		for (int i=14; i<20; i++)
        {
            phi(i) = 0.;
            for (int j=0; j<7; j++)
                phi(i) += gp(i-14,j)*y(j+7);
        }
        return phi;
    }

	const DenseVector &F(const Real &t, const DenseVector &y, const DenseVector &yprime) const
	{
        ff = Phi(t, y);
		ublas::noalias(ff) += ublas::prod(A(t,y,yprime), yprime);
		return ff;
	}

	const DenseMatrix &A(const Real &t, const DenseVector &y, const DenseVector &yprime) const 
    {
		using utility::sqr;

		const Real siph = sin(y(3));
		const Real siom = sin(y(5));
		const Real coth = cos(y(1));

		DenseMatrix m(7,7);
		m.clear();
		m(0,0) = m1*sqr(ra) + m2*(sqr(rr)-2*da*rr*coth+sqr(da)) + i1 + i2;
		m(1,0) = m2*(sqr(da)-da*rr*coth) + i2;
		m(1,1) = m2*sqr(da) + i2;
		m(2,2) = m3*(sqr(sa)+sqr(sb)) + i3;
		m(3,3) = m4*sqr(e-ea) + i4;
		m(4,3) = m4*(sqr(e-ea)+zt*(e-ea)*siph) + i4;
		m(4,4) = m4*(sqr(zt)+2*zt*(e-ea)*siph+sqr(e-ea)) + m5*(sqr(ta)+sqr(tb)) + i4 + i5;
		m(5,5) = m6*sqr(zf-fa) + i6;
		m(6,5) = m6*(sqr(zf-fa)-u*(zf-fa)*siom) + i6;
		m(6,6) = m6*(sqr(zf-fa)-2*u*(zf-fa)*siom+sqr(u)) + m7*(sqr(ua)+sqr(ub)) + i6 + i7;

		for (int j=1; j<7; j++)
			for (int i=0; i<j; i++)
				m(i,j) = m(j,i);

        for (int i=0; i<7; i++)
            MA(i,i) = -1.;
        for (int i=0; i<7; i++)
            for (int j=0; j<7; j++)
                MA(i+7,j+7) = m(i,j);

		return MA;
    }

	const DenseMatrix &B(const Real &t, const DenseVector &y, const DenseVector &yprime) const 
	{
		using utility::sqr;
		DenseMatrix &dfdy = MB;

		//-----------------------------------------------------------------------
		//     the Jacobian computed here is an approximation, see p. 540 of
		//     Hairer & Wanner 'solving ordinary differential equations II'
		//-----------------------------------------------------------------------
		const Real sibe = sin(y(0));
		const Real siga = sin(y(2));
		const Real siph = sin(y(3));
		const Real side = sin(y(4));
		const Real siom = sin(y(5));
		const Real siep = sin(y(6));

		const Real cobe = cos(y(0));
		const Real coth = cos(y(1));
		const Real coga = cos(y(2));
		const Real code = cos(y(4));
		const Real coep = cos(y(6));

		const Real sibeth = sin(y(0)+y(1));
		const Real siphde = sin(y(3)+y(4));
		const Real siomep = sin(y(5)+y(6));

		const Real cobeth = cos(y(0)+y(1));
		const Real cophde = cos(y(3)+y(4));
		const Real coomep = cos(y(5)+y(6));


		DenseMatrix m(7,7);
		m.clear();
		m(0,0) = m1*sqr(ra) + m2*(sqr(rr)-2*da*rr*coth+sqr(da)) + i1 + i2;
		m(1,0) = m2*(sqr(da)-da*rr*coth) + i2;
		m(1,1) = m2*sqr(da) + i2;
		m(2,2) = m3*(sqr(sa)+sqr(sb)) + i3;
		m(3,3) = m4*sqr(e-ea) + i4;
		m(4,3) = m4*(sqr(e-ea)+zt*(e-ea)*siph) + i4;
		m(4,4) = m4*(sqr(zt)+2*zt*(e-ea)*siph+sqr(e-ea)) + m5*(sqr(ta)+sqr(tb)) + i4 + i5;
		m(5,5) = m6*sqr(zf-fa) + i6;
		m(6,5) = m6*(sqr(zf-fa)-u*(zf-fa)*siom) + i6;
		m(6,6) = m6*(sqr(zf-fa)-2*u*(zf-fa)*siom+sqr(u)) + m7*(sqr(ua)+sqr(ub)) + i6 + i7;

		for (int j=1; j<7; j++)
			for (int i=0; i<j; i++)
				m(i,j) = m(j,i);

		DenseMatrix gp(6,7);
		gp.clear();
		gp(0,0) = - rr*sibe + d*sibeth;
		gp(0,1) = d*sibeth;
		gp(0,2) = - ss*coga;
		gp(1,0) = rr*cobe - d*cobeth;
		gp(1,1) = - d*cobeth;
		gp(1,2) = - ss*siga;
		gp(2,0) = - rr*sibe + d*sibeth;
		gp(2,1) = d*sibeth;
		gp(2,3) = - e*cophde;
		gp(2,4) = - e*cophde + zt*side;
		gp(3,0) = rr*cobe - d*cobeth;
		gp(3,1) = - d*cobeth;
		gp(3,3) = - e*siphde;
		gp(3,4) = - e*siphde - zt*code;
		gp(4,0) = - rr*sibe + d*sibeth;
		gp(4,1) = d*sibeth;
		gp(4,5) = zf*siomep;
		gp(4,6) = zf*siomep - u*coep;
		gp(5,0) = rr*cobe - d*cobeth;
		gp(5,1) = - d*cobeth;
		gp(5,5) = - zf*coomep;
		gp(5,6) = - zf*coomep - u*siep;


		for (int i=0; i<7; i++)
			dfdy(i,i+7) = 1.;
        for (int i=7; i<14; i++)
            for (int j=14; j<20; j++)
                dfdy(i,j) = gp(j-14,i-7);
        for (int i=14; i<20; i++)
            for (int j=7;j<14; j++)
                dfdy(i,j) = gp(i-14,j-7);

        if (1)
        { // this particularly ugly code was generated using Maple 
            Real Dphi2x1[7][7];
            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                    Dphi2x1[i][j] = 0.;
/*
            Dphi2x1[0][0] = (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[14] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[15] + (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[16] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[17] + (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[18] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[19];
            Dphi2x1[0][1] = m2 * da * rr * y[8] * (y[8] + 0.2e1 * y[7]) * cos(y[1]) + d * cos(y[0] + y[1]) * y[14] + d * sin(y[0] + y[1]) * y[15] + d * cos(y[0] + y[1]) * y[16] + d * sin(y[0] + y[1]) * y[17] + d * cos(y[0] + y[1]) * y[18] + d * sin(y[0] + y[1]) * y[19];
            Dphi2x1[1][0] = d * cos(y[0] + y[1]) * y[14] + d * sin(y[0] + y[1]) * y[15] + d * cos(y[0] + y[1]) * y[16] + d * sin(y[0] + y[1]) * y[17] + d * cos(y[0] + y[1]) * y[18] + d * sin(y[0] + y[1]) * y[19];
            Dphi2x1[1][1] = -m2 * da * rr * pow(y[7], 0.2e1) * cos(y[1]) + d * cos(y[0] + y[1]) * y[14] + d * sin(y[0] + y[1]) * y[15] + d * cos(y[0] + y[1]) * y[16] + d * sin(y[0] + y[1]) * y[17] + d * cos(y[0] + y[1]) * y[18] + d * sin(y[0] + y[1]) * y[19];
            Dphi2x1[2][2] = c0 / (pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) / 0.2e1 - c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.3e1 / 0.2e1) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) / 0.2e1 + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * pow(sc * cos(y[2]) - sd * sin(y[2]), 0.2e1) + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (-sc * sin(y[2]) - sd * cos(y[2])) + c0 / (pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2])) / 0.2e1 - c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.3e1 / 0.2e1) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2])) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) / 0.2e1 + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * pow(sd * cos(y[2]) + sc * sin(y[2]), 0.2e1) + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sc * cos(y[2]) - sd * sin(y[2])) + ss * sin(y[2]) * y[14] - ss * cos(y[2]) * y[15];
            Dphi2x1[3][3] = m4 * zt * (e - ea) * pow(y[11], 0.2e1) * sin(y[3]) + e * sin(y[3] + y[4]) * y[16] - e * cos(y[3] + y[4]) * y[17];
            Dphi2x1[3][4] = e * sin(y[3] + y[4]) * y[16] - e * cos(y[3] + y[4]) * y[17];
            Dphi2x1[4][3] = -m4 * zt * (e - ea) * y[10] * (y[10] + 0.2e1 * y[11]) * sin(y[3]) + e * sin(y[3] + y[4]) * y[16] - e * cos(y[3] + y[4]) * y[17];
            Dphi2x1[4][4] = (e * sin(y[3] + y[4]) + zt * cos(y[4])) * y[16] + (-e * cos(y[3] + y[4]) + zt * sin(y[4])) * y[17];
            Dphi2x1[5][5] = -m6 * u * (zf - fa) * pow(y[13], 0.2e1) * sin(y[5]) + zf * cos(y[5] + y[6]) * y[18] + zf * sin(y[5] + y[6]) * y[19];
            Dphi2x1[5][6] = zf * cos(y[5] + y[6]) * y[18] + zf * sin(y[5] + y[6]) * y[19];
            Dphi2x1[6][5] = m6 * u * (zf - fa) * y[12] * (y[12] + 0.2e1 * y[13]) * sin(y[5]) + zf * cos(y[5] + y[6]) * y[18] + zf * sin(y[5] + y[6]) * y[19];
            Dphi2x1[6][6] = (zf * cos(y[5] + y[6]) + u * sin(y[6])) * y[18] + (zf * sin(y[5] + y[6]) - u * cos(y[6])) * y[19];
*/
            Real Dphi2x2[7][7];
            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                    Dphi2x2[i][j] = 0.;
            Dphi2x2[0][0] = 0.2e1 * m2 * da * rr * y[8] * sin(y[1]);
            Dphi2x2[0][1] = m2 * da * rr * (y[8] + 0.2e1 * y[7]) * sin(y[1]) + m2 * da * rr * y[8] * sin(y[1]);
            Dphi2x2[1][0] = -0.2e1 * m2 * da * rr * y[7] * sin(y[1]);
            Dphi2x2[3][4] = -0.2e1 * m4 * zt * (e - ea) * y[11] * cos(y[3]);
            Dphi2x2[4][3] = m4 * zt * (e - ea) * (y[10] + 0.2e1 * y[11]) * cos(y[3]) + m4 * zt * (e - ea) * y[10] * cos(y[3]);
            Dphi2x2[4][4] = 0.2e1 * m4 * zt * (e - ea) * y[10] * cos(y[3]);
            Dphi2x2[5][6] = 0.2e1 * m6 * u * (zf - fa) * y[13] * cos(y[5]);
            Dphi2x2[6][5] = -m6 * u * (zf - fa) * (y[12] + 0.2e1 * y[13]) * cos(y[5]) - m6 * u * (zf - fa) * y[12] * cos(y[5]);
            Dphi2x2[6][6] = -0.2e1 * m6 * u * (zf - fa) * y[12] * cos(y[5]);

            Real Dphi3x1[6][7];
            for (int i=0; i<6; i++)
                for (int j=0; j<7; j++)
                    Dphi3x1[i][j] = 0.;
            Dphi3x1[0][0] = (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[0][1] = d * cos(y[0] + y[1]) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[0][2] = ss * sin(y[2]) * y[9];
            Dphi3x1[1][0] = (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[1][1] = d * sin(y[0] + y[1]) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[1][2] = -ss * cos(y[2]) * y[9];
            Dphi3x1[2][0] = (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[2][1] = d * cos(y[0] + y[1]) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[2][3] = e * sin(y[3] + y[4]) * y[10] + e * sin(y[3] + y[4]) * y[11];
            Dphi3x1[2][4] = e * sin(y[3] + y[4]) * y[10] + (e * sin(y[3] + y[4]) + zt * cos(y[4])) * y[11];
            Dphi3x1[3][0] = (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[3][1] = d * sin(y[0] + y[1]) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[3][3] = -e * cos(y[3] + y[4]) * y[10] - e * cos(y[3] + y[4]) * y[11];
            Dphi3x1[3][4] = -e * cos(y[3] + y[4]) * y[10] + (-e * cos(y[3] + y[4]) + zt * sin(y[4])) * y[11];
            Dphi3x1[4][0] = (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[4][1] = d * cos(y[0] + y[1]) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[4][5] = zf * cos(y[5] + y[6]) * y[12] + zf * cos(y[5] + y[6]) * y[13];
            Dphi3x1[4][6] = zf * cos(y[5] + y[6]) * y[12] + (zf * cos(y[5] + y[6]) + u * sin(y[6])) * y[13];
            Dphi3x1[5][0] = (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[5][1] = d * sin(y[0] + y[1]) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[5][5] = zf * sin(y[5] + y[6]) * y[12] + zf * sin(y[5] + y[6]) * y[13];
            Dphi3x1[5][6] = zf * sin(y[5] + y[6]) * y[12] + (zf * sin(y[5] + y[6]) - u * cos(y[6])) * y[13];

            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                {
                    dfdy(i+7,j) = Dphi2x1[i][j];
                    dfdy(i+7,j+7) = Dphi2x2[i][j];
                }
            for (int i=0; i<6; i++)
                for (int j=0; j<7; j++)
                    dfdy(i+14,j) = Dphi3x1[i][j];
            
        }
		return MB;
	}


	const DenseVector &initial() const
	{
		DenseVector &y = init_y;

		y(0)  = -0.0617138900142764496358948458001e0;
		y(1)  =  0e0;
		y(2)  =  0.455279819163070380255912382449e0;
		y(3)  =  0.222668390165885884674473185609e0;
		y(4)  =  0.487364979543842550225598953530e0;
		y(5)  = -0.222668390165885884674473185609e0;
		y(6)  =  1.23054744454982119249735015568e0;
		y(7)  =  0e0;
		y(8)  =  0e0;
		y(9)  =  0e0;
		y(10) =  0e0;
		y(11) =  0e0;
		y(12) =  0e0;
		y(13) =  0e0;
		y(14) =  98.5668703962410896057654982170e0;
		y(15) = -6.12268834425566265503114393122e0;
		y(16) =  0e0;
		y(17) =  0e0;
		y(18) =  0e0;
		y(19) =  0e0;

		return init_y;
	}

	const DenseVector &final() const
	{
		DenseVector &y = final_y;

		y(0)  = 15.8107711941657794;
		y(1)  = -15.7563710571773807;
		y(2)  = 0.0408222401100961;
		y(3)  = -0.5347301163587794;
		y(4)  = 0.5244099658794575;
		y(5)  = 0.5347301163587794;
		y(6)  = 1.0480807410416579;
		y(7)  = 1139.9203007108333168;
		y(8)  = -1424.3792932389776524;
		y(9)  = 11.0329116148355091;
		y(10) = 19.2933735871602501;
		y(11) = 0.5735698974635138;
		y(12) = -19.2933735869326171;
		y(13) = 0.3231791462295504;
		y(14) = 199.1753399486409251;
		y(15) = -29.7553083114498733;
		y(16) = 23.0665423222055033;
		y(17) = 31.4527159258740880;
		y(18) = 22.6424934101455264;
		y(19) = 11.6174053760077776;

		return final_y;
	}
};

struct ProblemAndrewsIndex2 : public ProblemBase, public ProblemAndrewsConstants
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector ff;
	mutable DenseVector init_y;
	mutable DenseVector final_y;
	mutable DenseVector init_yprime;

	ProblemAndrewsIndex2(const Real &T=3e-2) : ProblemBase(27,27, T)
	{
		ff.resize(m);
		init_y.resize(n);
		final_y.resize(n);
		init_yprime.resize(n);

		for (int i=0; i<14; i++)
			MA(i,i) = -1.0;
	}

	const DenseVector &F(const Real &t, const DenseVector &y, const DenseVector &yprime) const
	{
		using utility::sqr;

		const Real sibe = sin(y(0));
		const Real sith = sin(y(1));
		const Real siga = sin(y(2));
		const Real siph = sin(y(3));
		const Real side = sin(y(4));
		const Real siom = sin(y(5));
		const Real siep = sin(y(6));

		const Real cobe = cos(y(0));
		const Real coth = cos(y(1));
		const Real coga = cos(y(2));
		const Real coph = cos(y(3));
		const Real code = cos(y(4));
		const Real coom = cos(y(5));
		const Real coep = cos(y(6));

		const Real sibeth = sin(y(0)+y(1));
		const Real siphde = sin(y(3)+y(4));
		const Real siomep = sin(y(5)+y(6));

		const Real cobeth = cos(y(0)+y(1));
		const Real cophde = cos(y(3)+y(4));
		const Real coomep = cos(y(5)+y(6));

		const Real bep = y(7);
		const Real thp = y(8);
		const Real php = y(10);
		const Real dep = y(11);
		const Real omp = y(12);
		const Real epp = y(13);

		DenseMatrix m(7,7);
		m.clear();
		m(0,0) = m1*sqr(ra) + m2*(sqr(rr)-2*da*rr*coth+sqr(da)) + i1 + i2;
		m(1,0) = m2*(sqr(da)-da*rr*coth) + i2;
		m(1,1) = m2*sqr(da) + i2;
		m(2,2) = m3*(sqr(sa)+sqr(sb)) + i3;
		m(3,3) = m4*sqr(e-ea) + i4;
		m(4,3) = m4*(sqr(e-ea)+zt*(e-ea)*siph) + i4;
		m(4,4) = m4*(sqr(zt)+2*zt*(e-ea)*siph+sqr(e-ea)) + m5*(sqr(ta)+sqr(tb)) + i4 + i5;
		m(5,5) = m6*sqr(zf-fa) + i6;
		m(6,5) = m6*(sqr(zf-fa)-u*(zf-fa)*siom) + i6;
		m(6,6) = m6*(sqr(zf-fa)-2*u*(zf-fa)*siom+sqr(u)) + m7*(sqr(ua)+sqr(ub)) + i6 + i7;

		for (int j=1; j<7; j++)
			for (int i=0; i<j; i++)
				m(i,j) = m(j,i);

		const Real xd = sd*coga + sc*siga + xb;
		const Real yd = sd*siga - sc*coga + yb;
		const Real lang  = sqrt(sqr(xd-xc) + sqr(yd-yc));
		const Real force = - c0 * (lang - l0)/lang;
		const Real fx = force * (xd-xc);
		const Real fy = force * (yd-yc);
	
		DenseVector f(7);
		f(0) = mom - m2*da*rr*thp*(thp+2*bep)*sith;
		f(1) = m2*da*rr*sqr(bep)*sith;
		f(2) = fx*(sc*coga - sd*siga) + fy*(sd*coga + sc*siga);
		f(3) = m4*zt*(e-ea)*sqr(dep)*coph;
		f(4) = - m4*zt*(e-ea)*php*(php+2*dep)*coph;
		f(5) = - m6*u*(zf-fa)*sqr(epp)*coom;
		f(6) = m6*u*(zf-fa)*omp*(omp+2*epp)*coom;


		DenseMatrix gp(6,7);
		gp.clear();
		gp(0,0) = - rr*sibe + d*sibeth;
		gp(0,1) = d*sibeth;
		gp(0,2) = - ss*coga;
		gp(1,0) = rr*cobe - d*cobeth;
		gp(1,1) = - d*cobeth;
		gp(1,2) = - ss*siga;
		gp(2,0) = - rr*sibe + d*sibeth;
		gp(2,1) = d*sibeth;
		gp(2,3) = - e*cophde;
		gp(2,4) = - e*cophde + zt*side;
		gp(3,0) = rr*cobe - d*cobeth;
		gp(3,1) = - d*cobeth;
		gp(3,3) = - e*siphde;
		gp(3,4) = - e*siphde - zt*code;
		gp(4,0) = - rr*sibe + d*sibeth;
		gp(4,1) = d*sibeth;
		gp(4,5) = zf*siomep;
		gp(4,6) = zf*siomep - u*coep;
		gp(5,0) = rr*cobe - d*cobeth;
		gp(5,1) = - d*cobeth;
		gp(5,5) = - zf*coomep;
		gp(5,6) = - zf*coomep - u*siep;

		DenseVector g(6);
		g(0) = rr*cobe - d*cobeth - ss*siga - xb;
		g(1) = rr*sibe - d*sibeth + ss*coga - yb;
		g(2) = rr*cobe - d*cobeth - e*siphde - zt*code - xa;
		g(3) = rr*sibe - d*sibeth + e*cophde - zt*side - ya;
		g(4) = rr*cobe - d*cobeth - zf*coomep - u*siep - xa;
		g(5) = rr*sibe - d*sibeth - zf*siomep + u*coep - ya;

		for (int i=0; i<14; i++)
			ff(i) = y(i+7);

		for (int i=14; i<21; i++)
		{
			ff(i) = -f(i-14);
			for (int j=0; j<7; j++)
				ff(i) += m(i-14,j)*y(j+14);
			for (int j=0; j<6; j++)
				ff(i) += gp(j,i-14)*y(j+21);
		}
		for (int i=21; i<27; i++)
        {
            ff(i) = 0.;
            for (int j=0; j<7; j++)
                ff(i) += gp(i-21,j)*y(j+7);
        }

		// that is the only difference to the FORTRAN interface...	
		ublas::noalias(ff) += ublas::prod(MA, yprime);
		return ff;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &y, const DenseVector &yprime) const 
	{
		using utility::sqr;
		DenseMatrix &dfdy = MB;

		//-----------------------------------------------------------------------
		//     the Jacobian computed here is an approximation, see p. 540 of
		//     Hairer & Wanner 'solving ordinary differential equations II'
		//-----------------------------------------------------------------------

		const Real sibe = sin(y(0));
		const Real siga = sin(y(2));
		const Real siph = sin(y(3));
		const Real side = sin(y(4));
		const Real siom = sin(y(5));
		const Real siep = sin(y(6));

		const Real cobe = cos(y(0));
		const Real coth = cos(y(1));
		const Real coga = cos(y(2));
		const Real code = cos(y(4));
		const Real coep = cos(y(6));

		const Real sibeth = sin(y(0)+y(1));
		const Real siphde = sin(y(3)+y(4));
		const Real siomep = sin(y(5)+y(6));

		const Real cobeth = cos(y(0)+y(1));
		const Real cophde = cos(y(3)+y(4));
		const Real coomep = cos(y(5)+y(6));


		DenseMatrix m(7,7);
		m.clear();
		m(0,0) = m1*sqr(ra) + m2*(sqr(rr)-2*da*rr*coth+sqr(da)) + i1 + i2;
		m(1,0) = m2*(sqr(da)-da*rr*coth) + i2;
        m(0,1) = m(1,0);
		m(1,1) = m2*sqr(da) + i2;
		m(2,2) = m3*(sqr(sa)+sqr(sb)) + i3;
		m(3,3) = m4*sqr(e-ea) + i4;
		m(4,3) = m4*(sqr(e-ea)+zt*(e-ea)*siph) + i4;
        m(3,4) = m(4,3);
		m(4,4) = m4*(sqr(zt)+2*zt*(e-ea)*siph+sqr(e-ea)) + m5*(sqr(ta)+sqr(tb)) + i4 + i5;
		m(5,5) = m6*sqr(zf-fa) + i6;
		m(6,5) = m6*(sqr(zf-fa)-u*(zf-fa)*siom) + i6;
        m(5,6) = m(6,5);
		m(6,6) = m6*(sqr(zf-fa)-2*u*(zf-fa)*siom+sqr(u)) + m7*(sqr(ua)+sqr(ub)) + i6 + i7;

		DenseMatrix gp(6,7);
		gp.clear();
		gp(0,0) = - rr*sibe + d*sibeth;
		gp(0,1) = d*sibeth;
		gp(0,2) = - ss*coga;
		gp(1,0) = rr*cobe - d*cobeth;
		gp(1,1) = - d*cobeth;
		gp(1,2) = - ss*siga;
		gp(2,0) = - rr*sibe + d*sibeth;
		gp(2,1) = d*sibeth;
		gp(2,3) = - e*cophde;
		gp(2,4) = - e*cophde + zt*side;
		gp(3,0) = rr*cobe - d*cobeth;
		gp(3,1) = - d*cobeth;
		gp(3,3) = - e*siphde;
		gp(3,4) = - e*siphde - zt*code;
		gp(4,0) = - rr*sibe + d*sibeth;
		gp(4,1) = d*sibeth;
		gp(4,5) = zf*siomep;
		gp(4,6) = zf*siomep - u*coep;
		gp(5,0) = rr*cobe - d*cobeth;
		gp(5,1) = - d*cobeth;
		gp(5,5) = - zf*coomep;
		gp(5,6) = - zf*coomep - u*siep;


		for (int i=0; i<14; i++)
			dfdy(i,i+7) = 1e0;

		for (int i=0; i<7; i++)
			for (int j=0; j<7; j++)
				dfdy(14+j,14+i) = m(j,i);
				
		for (int i=0; i<6; i++)
			for (int j=0; j<7; j++)
				dfdy(14+j,21+i) = gp(i,j);

        
        if (0)
        { // this particularly ugly code was generated using Maple 

            // These are the missing terms that have been left out in the approximation from Hairer & Wanner.
            // Not surprisingly, they don't change much...
            Real Dq[7][7];
            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                    Dq[i][j] = 0.;
/*            
            Dq[0][0] = (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[21] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[22] + (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[23] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[24] + (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[25] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[26];
            Dq[0][1] = 0.2e1 * m2 * da * rr * sin(y[1]) * y[14] + m2 * da * rr * sin(y[1]) * y[15] + m2 * da * rr * y[8] * (y[8] + 0.2e1 * y[7]) * cos(y[1]) + d * cos(y[0] + y[1]) * y[21] + d * sin(y[0] + y[1]) * y[22] + d * cos(y[0] + y[1]) * y[23] + d * sin(y[0] + y[1]) * y[24] + d * cos(y[0] + y[1]) * y[25] + d * sin(y[0] + y[1]) * y[26];
            Dq[1][0] = d * cos(y[0] + y[1]) * y[21] + d * sin(y[0] + y[1]) * y[22] + d * cos(y[0] + y[1]) * y[23] + d * sin(y[0] + y[1]) * y[24] + d * cos(y[0] + y[1]) * y[25] + d * sin(y[0] + y[1]) * y[26];
            Dq[1][1] = m2 * da * rr * sin(y[1]) * y[14] - m2 * da * rr * pow(y[7], 0.2e1) * cos(y[1]) + d * cos(y[0] + y[1]) * y[21] + d * sin(y[0] + y[1]) * y[22] + d * cos(y[0] + y[1]) * y[23] + d * sin(y[0] + y[1]) * y[24] + d * cos(y[0] + y[1]) * y[25] + d * sin(y[0] + y[1]) * y[26];
            Dq[2][2] = c0 / (pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) 
                * (sc * cos(y[2]) - sd * sin(y[2])) / 0.2e1 - c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.3e1 / 0.2e1) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) / 0.2e1 + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * pow(sc * cos(y[2]) - sd * sin(y[2]), 0.2e1) + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (-sc * sin(y[2]) - sd * cos(y[2])) + c0 / (pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2])) / 0.2e1 - c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.3e1 / 0.2e1) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2])) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) / 0.2e1 + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * pow(sd * cos(y[2]) + sc * sin(y[2]), 0.2e1) + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sc * cos(y[2]) - sd * sin(y[2])) + ss * sin(y[2]) * y[21] - ss * cos(y[2]) * y[22];
            Dq[3][3] = m4 * zt * (e - ea) * cos(y[3]) * y[18] + m4 * zt * (e - ea) * pow(y[11], 0.2e1) * sin(y[3]) + e * sin(y[3] + y[4]) * y[23] - e * cos(y[3] + y[4]) * y[24];
            Dq[3][4] = e * sin(y[3] + y[4]) * y[23] - e * cos(y[3] + y[4]) * y[24];
            Dq[4][3] = m4 * zt * (e - ea) * cos(y[3]) * y[17] + 0.2e1 * m4 * zt * (e - ea) * cos(y[3]) * y[18] - m4 * zt * (e - ea) * y[10] * (y[10] + 0.2e1 * y[11]) * sin(y[3]) + e * sin(y[3] + y[4]) * y[23] - e * cos(y[3] + y[4]) * y[24];
            Dq[4][4] = (e * sin(y[3] + y[4]) + zt * cos(y[4])) * y[23] + (-e * cos(y[3] + y[4]) + zt * sin(y[4])) * y[24];
            Dq[5][5] = -m6 * u * (zf - fa) * cos(y[5]) * y[20] - m6 * u * (zf - fa) * pow(y[13], 0.2e1) * sin(y[5]) + zf * cos(y[5] + y[6]) * y[25] + zf * sin(y[5] + y[6]) * y[26];
            Dq[5][6] = zf * cos(y[5] + y[6]) * y[25] + zf * sin(y[5] + y[6]) * y[26];
            Dq[6][5] = -m6 * u * (zf - fa) * cos(y[5]) * y[19] - 0.2e1 * m6 * u * (zf - fa) * cos(y[5]) * y[20] + m6 * u * (zf - fa) * y[12] * (y[12] + 0.2e1 * y[13]) * sin(y[5]) + zf * cos(y[5] + y[6]) * y[25] + zf * sin(y[5] + y[6]) * y[26];
            Dq[6][6] = (zf * cos(y[5] + y[6]) + u * sin(y[6])) * y[25] + (zf * sin(y[5] + y[6]) - u * cos(y[6])) * y[26];
*/
            
            Real Dqp[7][7];
            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                    Dqp[i][j] = 0.;
            
            Dqp[0][0] = 0.2e1 * m2 * da * rr * y[8] * sin(y[1]);
            Dqp[0][1] = m2 * da * rr * (y[8] + 0.2e1 * y[7]) * sin(y[1]) + m2 * da * rr * y[8] * sin(y[1]);
            Dqp[1][0] = -0.2e1 * m2 * da * rr * y[7] * sin(y[1]);
            Dqp[3][4] = -0.2e1 * m4 * zt * (e - ea) * y[11] * cos(y[3]);
            Dqp[4][3] = m4 * zt * (e - ea) * (y[10] + 0.2e1 * y[11]) * cos(y[3]) + m4 * zt * (e - ea) * y[10] * cos(y[3]);
            Dqp[4][4] = 0.2e1 * m4 * zt * (e - ea) * y[10] * cos(y[3]);
            Dqp[5][6] = 0.2e1 * m6 * u * (zf - fa) * y[13] * cos(y[5]);
            Dqp[6][5] = -m6 * u * (zf - fa) * (y[12] + 0.2e1 * y[13]) * cos(y[5]) - m6 * u * (zf - fa) * y[12] * cos(y[5]);
            Dqp[6][6] = -0.2e1 * m6 * u * (zf - fa) * y[12] * cos(y[5]);
            
            
            Real Dphi3x1[6][7];
            for (int i=0; i<6; i++)
                for (int j=0; j<7; j++)
                    Dphi3x1[i][j] = 0.;
            Dphi3x1[0][0] = (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[0][1] = d * cos(y[0] + y[1]) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[0][2] = ss * sin(y[2]) * y[9];
            Dphi3x1[1][0] = (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[1][1] = d * sin(y[0] + y[1]) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[1][2] = -ss * cos(y[2]) * y[9];
            Dphi3x1[2][0] = (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[2][1] = d * cos(y[0] + y[1]) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[2][3] = e * sin(y[3] + y[4]) * y[10] + e * sin(y[3] + y[4]) * y[11];
            Dphi3x1[2][4] = e * sin(y[3] + y[4]) * y[10] + (e * sin(y[3] + y[4]) + zt * cos(y[4])) * y[11];
            Dphi3x1[3][0] = (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[3][1] = d * sin(y[0] + y[1]) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[3][3] = -e * cos(y[3] + y[4]) * y[10] - e * cos(y[3] + y[4]) * y[11];
            Dphi3x1[3][4] = -e * cos(y[3] + y[4]) * y[10] + (-e * cos(y[3] + y[4]) + zt * sin(y[4])) * y[11];
            Dphi3x1[4][0] = (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[4][1] = d * cos(y[0] + y[1]) * y[7] + d * cos(y[0] + y[1]) * y[8];
            Dphi3x1[4][5] = zf * cos(y[5] + y[6]) * y[12] + zf * cos(y[5] + y[6]) * y[13];
            Dphi3x1[4][6] = zf * cos(y[5] + y[6]) * y[12] + (zf * cos(y[5] + y[6]) + u * sin(y[6])) * y[13];
            Dphi3x1[5][0] = (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[5][1] = d * sin(y[0] + y[1]) * y[7] + d * sin(y[0] + y[1]) * y[8];
            Dphi3x1[5][5] = zf * sin(y[5] + y[6]) * y[12] + zf * sin(y[5] + y[6]) * y[13];
            Dphi3x1[5][6] = zf * sin(y[5] + y[6]) * y[12] + (zf * sin(y[5] + y[6]) - u * cos(y[6])) * y[13];

            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                {
                    dfdy(i+14,j) = Dq[i][j];
                    dfdy(i+14,j+7) = Dqp[i][j];
                }

            for (int i=0; i<7; i++)
                for (int j=0; j<6; j++)
                    dfdy(21+j,i) = Dphi3x1[j][i];
        }



		return MB;
	}


	const DenseVector &initial() const
	{
		DenseVector &y = init_y;

		y(0)  = -0.0617138900142764496358948458001e0;
		y(1)  =  0e0;
		y(2)  =  0.455279819163070380255912382449e0;
		y(3)  =  0.222668390165885884674473185609e0;
		y(4)  =  0.487364979543842550225598953530e0;
		y(5)  = -0.222668390165885884674473185609e0;
		y(6)  =  1.23054744454982119249735015568e0;
		y(7)  =  0e0;
		y(8)  =  0e0;
		y(9)  =  0e0;
		y(10) =  0e0;
		y(11) =  0e0;
		y(12) =  0e0;
		y(13) =  0e0;
		y(14) =  14222.4439199541138705911625887e0;
		y(15) = -10666.8329399655854029433719415e0;
		y(16) =  0e0;
		y(17) =  0e0;
		y(18) =  0e0;
		y(19) =  0e0;
		y(20) =  0e0;
		y(21) =  98.5668703962410896057654982170e0;
		y(23) = -6.12268834425566265503114393122e0;
		y(23) =  0e0;
		y(24) =  0e0;
		y(25) =  0e0;
		y(26) =  0e0;

		return init_y;
	}

	const DenseVector &final() const
	{
		DenseVector &y = final_y;

		y(0)  = 15.8107711941657794;
		y(1)  = -15.7563710571773807;
		y(2)  = 0.0408222401100961;
		y(3)  = -0.5347301163587794;
		y(4)  = 0.5244099658794575;
		y(5)  = 0.5347301163587794;
		y(6)  = 1.0480807410416579;
		y(7)  = 1139.9203007108333168;
		y(8)  = -1424.3792932389776524;
		y(9)  = 11.0329116148355091;
		y(10) = 19.2933735871602501;
		y(11) = 0.5735698974635138;
		y(12) = -19.2933735869326171;
		y(13) = 0.3231791462295504;
		y(14) = -24631.7008203593468352;
		y(15) = 51850.2920066463047988;
		y(16) = 324102.5832561646238901;
		y(17) = 566749.3904016350861639;
		y(18) = 16743.6304924886462686;
		y(19) = -566749.3894865390611812;
		y(20) = 9826.5193467981262074;
		y(21) = 199.1753399486409251;
		y(23) = -29.7553083114498733;
		y(23) = 23.0665423222055033;
		y(24) = 31.4527159258740880;
		y(25) = 22.6424934101455264;
		y(26) = 11.6174053760077776;

		return final_y;
	}

	const DenseVector &initial_deriv() const
	{
		initial();
		for (int i=0; i<14; i++)
			init_yprime(i) = init_y(i+7);
		for (int i=14; i<27; i++)
			init_yprime(i) = 0e0;
		return init_yprime;
	}
};










struct ProblemAndrewsIndex1Small : public ProblemBase, public ProblemAndrewsConstants
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector ff;
	mutable DenseVector init_y;
	mutable DenseVector final_y;

	ProblemAndrewsIndex1Small(const Real &T=3e-2) : ProblemBase(20,20, T)
	{
		ff.resize(m);
		init_y.resize(n);
		final_y.resize(n);
	}

	const DenseVector &F(const Real &t, const DenseVector &y, const DenseVector &yprime) const
	{
		using utility::sqr;

		const Real sibe = sin(y(0));
		const Real sith = sin(y(1));
		const Real siga = sin(y(2));
		const Real siph = sin(y(3));
		const Real side = sin(y(4));
		const Real siom = sin(y(5));
		const Real siep = sin(y(6));

		const Real cobe = cos(y(0));
		const Real coth = cos(y(1));
		const Real coga = cos(y(2));
		const Real coph = cos(y(3));
		const Real code = cos(y(4));
		const Real coom = cos(y(5));
		const Real coep = cos(y(6));

		const Real sibeth = sin(y(0)+y(1));
		const Real siphde = sin(y(3)+y(4));
		const Real siomep = sin(y(5)+y(6));

		const Real cobeth = cos(y(0)+y(1));
		const Real cophde = cos(y(3)+y(4));
		const Real coomep = cos(y(5)+y(6));

		const Real bep = y(7);
		const Real thp = y(8);
		const Real php = y(10);
		const Real dep = y(11);
		const Real omp = y(12);
		const Real epp = y(13);

		DenseMatrix m(7,7);
		m.clear();
		m(0,0) = m1*sqr(ra) + m2*(sqr(rr)-2*da*rr*coth+sqr(da)) + i1 + i2;
		m(1,0) = m2*(sqr(da)-da*rr*coth) + i2;
		m(1,1) = m2*sqr(da) + i2;
		m(2,2) = m3*(sqr(sa)+sqr(sb)) + i3;
		m(3,3) = m4*sqr(e-ea) + i4;
		m(4,3) = m4*(sqr(e-ea)+zt*(e-ea)*siph) + i4;
		m(4,4) = m4*(sqr(zt)+2*zt*(e-ea)*siph+sqr(e-ea)) + m5*(sqr(ta)+sqr(tb)) + i4 + i5;
		m(5,5) = m6*sqr(zf-fa) + i6;
		m(6,5) = m6*(sqr(zf-fa)-u*(zf-fa)*siom) + i6;
		m(6,6) = m6*(sqr(zf-fa)-2*u*(zf-fa)*siom+sqr(u)) + m7*(sqr(ua)+sqr(ub)) + i6 + i7;

		for (int j=1; j<7; j++)
			for (int i=0; i<j; i++)
				m(i,j) = m(j,i);

		const Real xd = sd*coga + sc*siga + xb;
		const Real yd = sd*siga - sc*coga + yb;
		const Real lang  = sqrt(sqr(xd-xc) + sqr(yd-yc));
		const Real force = - c0 * (lang - l0)/lang;
		const Real fx = force * (xd-xc);
		const Real fy = force * (yd-yc);
	
		DenseVector f(7);
		f(0) = mom - m2*da*rr*thp*(thp+2*bep)*sith;
		f(1) = m2*da*rr*sqr(bep)*sith;
		f(2) = fx*(sc*coga - sd*siga) + fy*(sd*coga + sc*siga);
		f(3) = m4*zt*(e-ea)*sqr(dep)*coph;
		f(4) = - m4*zt*(e-ea)*php*(php+2*dep)*coph;
		f(5) = - m6*u*(zf-fa)*sqr(epp)*coom;
		f(6) = m6*u*(zf-fa)*omp*(omp+2*epp)*coom;


		DenseMatrix gp(6,7);
		gp.clear();
		gp(0,0) = - rr*sibe + d*sibeth;
		gp(0,1) = d*sibeth;
		gp(0,2) = - ss*coga;
		gp(1,0) = rr*cobe - d*cobeth;
		gp(1,1) = - d*cobeth;
		gp(1,2) = - ss*siga;
		gp(2,0) = - rr*sibe + d*sibeth;
		gp(2,1) = d*sibeth;
		gp(2,3) = - e*cophde;
		gp(2,4) = - e*cophde + zt*side;
		gp(3,0) = rr*cobe - d*cobeth;
		gp(3,1) = - d*cobeth;
		gp(3,3) = - e*siphde;
		gp(3,4) = - e*siphde - zt*code;
		gp(4,0) = - rr*sibe + d*sibeth;
		gp(4,1) = d*sibeth;
		gp(4,5) = zf*siomep;
		gp(4,6) = zf*siomep - u*coep;
		gp(5,0) = rr*cobe - d*cobeth;
		gp(5,1) = - d*cobeth;
		gp(5,5) = - zf*coomep;
		gp(5,6) = - zf*coomep - u*siep;

        DenseVector gpp(6);
        gpp(0) = -rr*cobe*sqr(y(7+0)) + d*cobeth*sqr(y(7+0)+y(7+1)) + ss*siga*sqr(y(7+2));
        gpp(1) = -rr*sibe*sqr(y(7+0)) + d*sibeth*sqr(y(7+0)+y(7+1)) - ss*coga*sqr(y(7+2));
        gpp(2) = -rr*cobe*sqr(y(7+0)) + d*cobeth*sqr(y(7+0)+y(7+1)) + e*siphde*sqr(y(7+3)+y(7+4)) + zt*code*sqr(y(7+4));
        gpp(3) = -rr*sibe*sqr(y(7+0)) + d*sibeth*sqr(y(7+0)+y(7+1)) - e*cophde*sqr(y(7+3)+y(7+4)) + zt*side*sqr(y(7+4));
        gpp(4) = -rr*cobe*sqr(y(7+0)) + d*cobeth*sqr(y(7+0)+y(7+1)) + zf*coomep*sqr(y(7+5)+y(7+6)) + u*siep*sqr(y(7+6));
        gpp(5) = -rr*sibe*sqr(y(7+0)) + d*sibeth*sqr(y(7+0)+y(7+1)) + zf*siomep*sqr(y(7+5)+y(7+6)) - u*coep*sqr(y(7+6));


		for (int i=0; i<7; i++)
			ff(i) = y(i+7);

		for (int i=7; i<14; i++)
		{
			ff(i) = -f(i-7);
			for (int j=0; j<6; j++)
				ff(i) += gp(j,i-7)*y(j+14);
		}
		for (int i=14; i<20; i++)
            ff(i) = gpp(i-14);

		ublas::noalias(ff) += ublas::prod(A(t,y,yprime), yprime);
		return ff;
	}

	const DenseMatrix &A(const Real &t, const DenseVector &y, const DenseVector &yprime) const 
    {
		using utility::sqr;

		const Real sibe = sin(y(0));
		const Real siga = sin(y(2));
		const Real siph = sin(y(3));
		const Real side = sin(y(4));
		const Real siom = sin(y(5));
		const Real siep = sin(y(6));

		const Real cobe = cos(y(0));
		const Real coth = cos(y(1));
		const Real coga = cos(y(2));
		const Real code = cos(y(4));
		const Real coep = cos(y(6));

		DenseMatrix m(7,7);
		m.clear();
		m(0,0) = m1*sqr(ra) + m2*(sqr(rr)-2*da*rr*coth+sqr(da)) + i1 + i2;
		m(1,0) = m2*(sqr(da)-da*rr*coth) + i2;
        m(0,1) = m(1,0);
		m(1,1) = m2*sqr(da) + i2;
		m(2,2) = m3*(sqr(sa)+sqr(sb)) + i3;
		m(3,3) = m4*sqr(e-ea) + i4;
		m(4,3) = m4*(sqr(e-ea)+zt*(e-ea)*siph) + i4;
        m(3,4) = m(4,3);
		m(4,4) = m4*(sqr(zt)+2*zt*(e-ea)*siph+sqr(e-ea)) + m5*(sqr(ta)+sqr(tb)) + i4 + i5;
		m(5,5) = m6*sqr(zf-fa) + i6;
		m(6,5) = m6*(sqr(zf-fa)-u*(zf-fa)*siom) + i6;
        m(5,6) = m(6,5);
		m(6,6) = m6*(sqr(zf-fa)-2*u*(zf-fa)*siom+sqr(u)) + m7*(sqr(ua)+sqr(ub)) + i6 + i7;

		const Real sibeth = sin(y(0)+y(1));
		const Real siphde = sin(y(3)+y(4));
		const Real siomep = sin(y(5)+y(6));

		const Real cobeth = cos(y(0)+y(1));
		const Real cophde = cos(y(3)+y(4));
		const Real coomep = cos(y(5)+y(6));

		DenseMatrix gp(6,7);
		gp.clear();
		gp(0,0) = - rr*sibe + d*sibeth;
		gp(0,1) = d*sibeth;
		gp(0,2) = - ss*coga;
		gp(1,0) = rr*cobe - d*cobeth;
		gp(1,1) = - d*cobeth;
		gp(1,2) = - ss*siga;
		gp(2,0) = - rr*sibe + d*sibeth;
		gp(2,1) = d*sibeth;
		gp(2,3) = - e*cophde;
		gp(2,4) = - e*cophde + zt*side;
		gp(3,0) = rr*cobe - d*cobeth;
		gp(3,1) = - d*cobeth;
		gp(3,3) = - e*siphde;
		gp(3,4) = - e*siphde - zt*code;
		gp(4,0) = - rr*sibe + d*sibeth;
		gp(4,1) = d*sibeth;
		gp(4,5) = zf*siomep;
		gp(4,6) = zf*siomep - u*coep;
		gp(5,0) = rr*cobe - d*cobeth;
		gp(5,1) = - d*cobeth;
		gp(5,5) = - zf*coomep;
		gp(5,6) = - zf*coomep - u*siep;


        for (int i=0; i<7; i++)
            MA(i,i) = -1.;
        for (int i=0; i<7; i++)
            for (int j=0; j<7; j++)
                MA(i+7,j+7) = m(i,j);

        for (int i=0; i<6; i++)
            for (int j=0; j<7; j++)
                MA(i+14,j+7) = gp(i,j);

		return MA;
    }

	const DenseMatrix &B(const Real &t, const DenseVector &y, const DenseVector &yprime) const 
	{
		using utility::sqr;
		DenseMatrix &dfdy = MB;

		//-----------------------------------------------------------------------
		//     the Jacobian computed here is an approximation, see p. 540 of
		//     Hairer & Wanner 'solving ordinary differential equations II'
		//-----------------------------------------------------------------------

		const Real sibe = sin(y(0));
		const Real siga = sin(y(2));
		const Real siph = sin(y(3));
		const Real side = sin(y(4));
		const Real siom = sin(y(5));
		const Real siep = sin(y(6));

		const Real cobe = cos(y(0));
		const Real coth = cos(y(1));
		const Real coga = cos(y(2));
		const Real code = cos(y(4));
		const Real coep = cos(y(6));

		const Real sibeth = sin(y(0)+y(1));
		const Real siphde = sin(y(3)+y(4));
		const Real siomep = sin(y(5)+y(6));

		const Real cobeth = cos(y(0)+y(1));
		const Real cophde = cos(y(3)+y(4));
		const Real coomep = cos(y(5)+y(6));


		DenseMatrix m(7,7);
		m.clear();
		m(0,0) = m1*sqr(ra) + m2*(sqr(rr)-2*da*rr*coth+sqr(da)) + i1 + i2;
		m(1,0) = m2*(sqr(da)-da*rr*coth) + i2;
        m(0,1) = m(1,0);
		m(1,1) = m2*sqr(da) + i2;
		m(2,2) = m3*(sqr(sa)+sqr(sb)) + i3;
		m(3,3) = m4*sqr(e-ea) + i4;
		m(4,3) = m4*(sqr(e-ea)+zt*(e-ea)*siph) + i4;
        m(3,4) = m(4,3);
		m(4,4) = m4*(sqr(zt)+2*zt*(e-ea)*siph+sqr(e-ea)) + m5*(sqr(ta)+sqr(tb)) + i4 + i5;
		m(5,5) = m6*sqr(zf-fa) + i6;
		m(6,5) = m6*(sqr(zf-fa)-u*(zf-fa)*siom) + i6;
        m(5,6) = m(6,5);
		m(6,6) = m6*(sqr(zf-fa)-2*u*(zf-fa)*siom+sqr(u)) + m7*(sqr(ua)+sqr(ub)) + i6 + i7;

		DenseMatrix gp(6,7);
		gp.clear();
		gp(0,0) = - rr*sibe + d*sibeth;
		gp(0,1) = d*sibeth;
		gp(0,2) = - ss*coga;
		gp(1,0) = rr*cobe - d*cobeth;
		gp(1,1) = - d*cobeth;
		gp(1,2) = - ss*siga;
		gp(2,0) = - rr*sibe + d*sibeth;
		gp(2,1) = d*sibeth;
		gp(2,3) = - e*cophde;
		gp(2,4) = - e*cophde + zt*side;
		gp(3,0) = rr*cobe - d*cobeth;
		gp(3,1) = - d*cobeth;
		gp(3,3) = - e*siphde;
		gp(3,4) = - e*siphde - zt*code;
		gp(4,0) = - rr*sibe + d*sibeth;
		gp(4,1) = d*sibeth;
		gp(4,5) = zf*siomep;
		gp(4,6) = zf*siomep - u*coep;
		gp(5,0) = rr*cobe - d*cobeth;
		gp(5,1) = - d*cobeth;
		gp(5,5) = - zf*coomep;
		gp(5,6) = - zf*coomep - u*siep;


		for (int i=0; i<7; i++)
			dfdy(i,i+7) = 1.;
        for (int i=7; i<14; i++)
            for (int j=14; j<20; j++)
                dfdy(i,j) = gp(j-14,i-7);
        for (int i=14; i<20; i++)
            for (int j=7;j<14; j++)
                dfdy(i,j) = gp(i-14,j-7);

        if (1)
        { // this particularly ugly code was generated using Maple 
            Real Dphi2x1[7][7];
            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                    Dphi2x1[i][j] = 0.;
/*
            Dphi2x1[0][0] = (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[14] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[15] + (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[16] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[17] + (-rr * cos(y[0]) + d * cos(y[0] + y[1])) * y[18] + (-rr * sin(y[0]) + d * sin(y[0] + y[1])) * y[19];
            Dphi2x1[0][1] = m2 * da * rr * y[8] * (y[8] + 0.2e1 * y[7]) * cos(y[1]) + d * cos(y[0] + y[1]) * y[14] + d * sin(y[0] + y[1]) * y[15] + d * cos(y[0] + y[1]) * y[16] + d * sin(y[0] + y[1]) * y[17] + d * cos(y[0] + y[1]) * y[18] + d * sin(y[0] + y[1]) * y[19];
            Dphi2x1[1][0] = d * cos(y[0] + y[1]) * y[14] + d * sin(y[0] + y[1]) * y[15] + d * cos(y[0] + y[1]) * y[16] + d * sin(y[0] + y[1]) * y[17] + d * cos(y[0] + y[1]) * y[18] + d * sin(y[0] + y[1]) * y[19];
            Dphi2x1[1][1] = -m2 * da * rr * pow(y[7], 0.2e1) * cos(y[1]) + d * cos(y[0] + y[1]) * y[14] + d * sin(y[0] + y[1]) * y[15] + d * cos(y[0] + y[1]) * y[16] + d * sin(y[0] + y[1]) * y[17] + d * cos(y[0] + y[1]) * y[18] + d * sin(y[0] + y[1]) * y[19];
            Dphi2x1[2][2] = c0 / (pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) / 0.2e1 - c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.3e1 / 0.2e1) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) / 0.2e1 + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * pow(sc * cos(y[2]) - sd * sin(y[2]), 0.2e1) + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (-sc * sin(y[2]) - sd * cos(y[2])) + c0 / (pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2])) / 0.2e1 - c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.3e1 / 0.2e1) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2])) * (0.2e1 * (sd * cos(y[2]) + sc * sin(y[2]) + xb - xc) * (sc * cos(y[2]) - sd * sin(y[2])) + 0.2e1 * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sd * cos(y[2]) + sc * sin(y[2]))) / 0.2e1 + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * pow(sd * cos(y[2]) + sc * sin(y[2]), 0.2e1) + c0 * (sqrt(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1)) - l0) * pow(pow(sd * cos(y[2]) + sc * sin(y[2]) + xb - xc, 0.2e1) + pow(sd * sin(y[2]) - sc * cos(y[2]) + yb - yc, 0.2e1), -0.1e1 / 0.2e1) * (sd * sin(y[2]) - sc * cos(y[2]) + yb - yc) * (sc * cos(y[2]) - sd * sin(y[2])) + ss * sin(y[2]) * y[14] - ss * cos(y[2]) * y[15];
            Dphi2x1[3][3] = m4 * zt * (e - ea) * pow(y[11], 0.2e1) * sin(y[3]) + e * sin(y[3] + y[4]) * y[16] - e * cos(y[3] + y[4]) * y[17];
            Dphi2x1[3][4] = e * sin(y[3] + y[4]) * y[16] - e * cos(y[3] + y[4]) * y[17];
            Dphi2x1[4][3] = -m4 * zt * (e - ea) * y[10] * (y[10] + 0.2e1 * y[11]) * sin(y[3]) + e * sin(y[3] + y[4]) * y[16] - e * cos(y[3] + y[4]) * y[17];
            Dphi2x1[4][4] = (e * sin(y[3] + y[4]) + zt * cos(y[4])) * y[16] + (-e * cos(y[3] + y[4]) + zt * sin(y[4])) * y[17];
            Dphi2x1[5][5] = -m6 * u * (zf - fa) * pow(y[13], 0.2e1) * sin(y[5]) + zf * cos(y[5] + y[6]) * y[18] + zf * sin(y[5] + y[6]) * y[19];
            Dphi2x1[5][6] = zf * cos(y[5] + y[6]) * y[18] + zf * sin(y[5] + y[6]) * y[19];
            Dphi2x1[6][5] = m6 * u * (zf - fa) * y[12] * (y[12] + 0.2e1 * y[13]) * sin(y[5]) + zf * cos(y[5] + y[6]) * y[18] + zf * sin(y[5] + y[6]) * y[19];
            Dphi2x1[6][6] = (zf * cos(y[5] + y[6]) + u * sin(y[6])) * y[18] + (zf * sin(y[5] + y[6]) - u * cos(y[6])) * y[19];
*/
            Real Dphi2x2[7][7];
            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                    Dphi2x2[i][j] = 0.;
            
            Dphi2x2[0][0] = 0.2e1 * m2 * da * rr * y[8] * sin(y[1]);
            Dphi2x2[0][1] = m2 * da * rr * (y[8] + 0.2e1 * y[7]) * sin(y[1]) + m2 * da * rr * y[8] * sin(y[1]);
            Dphi2x2[1][0] = -0.2e1 * m2 * da * rr * y[7] * sin(y[1]);
            Dphi2x2[3][4] = -0.2e1 * m4 * zt * (e - ea) * y[11] * cos(y[3]);
            Dphi2x2[4][3] = m4 * zt * (e - ea) * (y[10] + 0.2e1 * y[11]) * cos(y[3]) + m4 * zt * (e - ea) * y[10] * cos(y[3]);
            Dphi2x2[4][4] = 0.2e1 * m4 * zt * (e - ea) * y[10] * cos(y[3]);
            Dphi2x2[5][6] = 0.2e1 * m6 * u * (zf - fa) * y[13] * cos(y[5]);
            Dphi2x2[6][5] = -m6 * u * (zf - fa) * (y[12] + 0.2e1 * y[13]) * cos(y[5]) - m6 * u * (zf - fa) * y[12] * cos(y[5]);
            Dphi2x2[6][6] = -0.2e1 * m6 * u * (zf - fa) * y[12] * cos(y[5]);

            Real Dphi3x1[6][7];
            for (int i=0; i<6; i++)
                for (int j=0; j<7; j++)
                    Dphi3x1[i][j] = 0.;

            Dphi3x1[0][0] = rr * sin(y[0]) * pow(y[7], 0.2e1) - d * sin(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[0][1] = -d * sin(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[0][2] = ss * cos(y[2]) * pow(y[9], 0.2e1);
            Dphi3x1[1][0] = -rr * cos(y[0]) * pow(y[7], 0.2e1) + d * cos(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[1][1] = d * cos(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[1][2] = ss * sin(y[2]) * pow(y[9], 0.2e1);
            Dphi3x1[2][0] = rr * sin(y[0]) * pow(y[7], 0.2e1) - d * sin(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[2][1] = -d * sin(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[2][3] = e * cos(y[3] + y[4]) * pow(y[10] + y[11], 0.2e1);
            Dphi3x1[2][4] = e * cos(y[3] + y[4]) * pow(y[10] + y[11], 0.2e1) - zt * sin(y[4]) * pow(y[11], 0.2e1);
            Dphi3x1[3][0] = -rr * cos(y[0]) * pow(y[7], 0.2e1) + d * cos(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[3][1] = d * cos(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[3][3] = e * sin(y[3] + y[4]) * pow(y[10] + y[11], 0.2e1);
            Dphi3x1[3][4] = e * sin(y[3] + y[4]) * pow(y[10] + y[11], 0.2e1) + zt * cos(y[4]) * pow(y[11], 0.2e1);
            Dphi3x1[4][0] = rr * sin(y[0]) * pow(y[7], 0.2e1) - d * sin(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[4][1] = -d * sin(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[4][5] = -zf * sin(y[5] + y[6]) * pow(y[12] + y[13], 0.2e1);
            Dphi3x1[4][6] = -zf * sin(y[5] + y[6]) * pow(y[12] + y[13], 0.2e1) + u * cos(y[6]) * pow(y[13], 0.2e1);
            Dphi3x1[5][0] = -rr * cos(y[0]) * pow(y[7], 0.2e1) + d * cos(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[5][1] = d * cos(y[0] + y[1]) * pow(y[7] + y[8], 0.2e1);
            Dphi3x1[5][5] = zf * cos(y[5] + y[6]) * pow(y[12] + y[13], 0.2e1);
            Dphi3x1[5][6] = zf * cos(y[5] + y[6]) * pow(y[12] + y[13], 0.2e1) + u * sin(y[6]) * pow(y[13], 0.2e1);


            Real Dphi3x2[6][7];
            for (int i=0; i<6; i++)
                for (int j=0; j<7; j++)
                    Dphi3x2[i][j] = 0.;

            Dphi3x2[0][0] = -0.2e1 * rr * cos(y[0]) * y[7] + 0.2e1 * d * cos(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[0][1] = 0.2e1 * d * cos(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[0][2] = 0.2e1 * ss * sin(y[2]) * y[9];
            Dphi3x2[1][0] = -0.2e1 * rr * sin(y[0]) * y[7] + 0.2e1 * d * sin(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[1][1] = 0.2e1 * d * sin(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[1][2] = -0.2e1 * ss * cos(y[2]) * y[9];
            Dphi3x2[2][0] = -0.2e1 * rr * cos(y[0]) * y[7] + 0.2e1 * d * cos(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[2][1] = 0.2e1 * d * cos(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[2][3] = 0.2e1 * e * sin(y[3] + y[4]) * (y[10] + y[11]);
            Dphi3x2[2][4] = 0.2e1 * e * sin(y[3] + y[4]) * (y[10] + y[11]) + 0.2e1 * zt * cos(y[4]) * y[11];
            Dphi3x2[3][0] = -0.2e1 * rr * sin(y[0]) * y[7] + 0.2e1 * d * sin(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[3][1] = 0.2e1 * d * sin(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[3][3] = -0.2e1 * e * cos(y[3] + y[4]) * (y[10] + y[11]);
            Dphi3x2[3][4] = -0.2e1 * e * cos(y[3] + y[4]) * (y[10] + y[11]) + 0.2e1 * zt * sin(y[4]) * y[11];
            Dphi3x2[4][0] = -0.2e1 * rr * cos(y[0]) * y[7] + 0.2e1 * d * cos(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[4][1] = 0.2e1 * d * cos(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[4][5] = 0.2e1 * zf * cos(y[5] + y[6]) * (y[12] + y[13]);
            Dphi3x2[4][6] = 0.2e1 * zf * cos(y[5] + y[6]) * (y[12] + y[13]) + 0.2e1 * u * sin(y[6]) * y[13];
            Dphi3x2[5][0] = -0.2e1 * rr * sin(y[0]) * y[7] + 0.2e1 * d * sin(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[5][1] = 0.2e1 * d * sin(y[0] + y[1]) * (y[7] + y[8]);
            Dphi3x2[5][5] = 0.2e1 * zf * sin(y[5] + y[6]) * (y[12] + y[13]);
            Dphi3x2[5][6] = 0.2e1 * zf * sin(y[5] + y[6]) * (y[12] + y[13]) - 0.2e1 * u * cos(y[6]) * y[13];
            

            for (int i=0; i<7; i++)
                for (int j=0; j<7; j++)
                {
                    dfdy(i+7,j) = Dphi2x1[i][j];
                    dfdy(i+7,j+7) = Dphi2x2[i][j];
                }
            for (int i=0; i<6; i++)
                for (int j=0; j<7; j++)
                {
                    dfdy(i+14,j) = Dphi3x1[i][j];
                    dfdy(i+14,j+7) = Dphi3x2[i][j];
                }
            
        }
		return MB;
	}


	const DenseVector &initial() const
	{
		DenseVector &y = init_y;

		y(0)  = -0.0617138900142764496358948458001e0;
		y(1)  =  0e0;
		y(2)  =  0.455279819163070380255912382449e0;
		y(3)  =  0.222668390165885884674473185609e0;
		y(4)  =  0.487364979543842550225598953530e0;
		y(5)  = -0.222668390165885884674473185609e0;
		y(6)  =  1.23054744454982119249735015568e0;
		y(7)  =  0e0;
		y(8)  =  0e0;
		y(9)  =  0e0;
		y(10) =  0e0;
		y(11) =  0e0;
		y(12) =  0e0;
		y(13) =  0e0;
		y(14) =  98.5668703962410896057654982170e0;
		y(15) = -6.12268834425566265503114393122e0;
		y(16) =  0e0;
		y(17) =  0e0;
		y(18) =  0e0;
		y(19) =  0e0;

		return init_y;
	}

	const DenseVector &final() const
	{
		DenseVector &y = final_y;

		y(0)  = 15.8107711941657794;
		y(1)  = -15.7563710571773807;
		y(2)  = 0.0408222401100961;
		y(3)  = -0.5347301163587794;
		y(4)  = 0.5244099658794575;
		y(5)  = 0.5347301163587794;
		y(6)  = 1.0480807410416579;
		y(7)  = 1139.9203007108333168;
		y(8)  = -1424.3792932389776524;
		y(9)  = 11.0329116148355091;
		y(10) = 19.2933735871602501;
		y(11) = 0.5735698974635138;
		y(12) = -19.2933735869326171;
		y(13) = 0.3231791462295504;
		y(14) = 199.1753399486409251;
		y(15) = -29.7553083114498733;
		y(16) = 23.0665423222055033;
		y(17) = 31.4527159258740880;
		y(18) = 22.6424934101455264;
		y(19) = 11.6174053760077776;

		return final_y;
	}
};



/*
Reference:
W. C. Rheinboldt, University of Pittsburgh
sample programs for dae_solve.tgz
http://www.netlib.org/ode/daesolve/

!-----------------------------------------------------------------
! Driver for applying the DAE-solver with daen1 for solving
! the "figure 8" problem
!
!   u^2 + u'^2 - 1  = 0
!   2*u*u' - w      = 0 
!   u(0) = 0, u'(0) = 1, w(0) = 0
!
! with the exact solution
!
!   u(t) = sin t, w(t) = sin 2t
!
! Note that here the (nu + nw) x (nu + nw) matrix (D_pF, D_wF)
! has the form
!                 (2*p    0)
!                 (2*u   -1)
! and hence is nonsingular only as long as  p .ne. 0. This does
! not hold for t = k*pi/2. These points are removable 
! singularities of the solution curve which the code passes 
! easily but where dassl fails.
!-----------------------------------------------------------------
*/
struct ProblemFigureEight : public ProblemBase
{

	typedef problem_nonlinear problem_type;

	mutable DenseVector res;
    mutable DenseVector init_y;
    mutable DenseVector init_yprime;

	ProblemFigureEight() : ProblemBase(2,2, 1.)
	{
		assert(n == m);
		res.resize(n);
        init_y.resize(n);
        init_yprime.resize(n);
	}


	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
        using utility::sqr;
		init(res) = 
            sqr(u(0)) + sqr(Du(0)) - 1.,
            2.*u(0)*Du(0) - u(1);
		return res;
	}

	const DenseMatrix &A(const Real &t, const DenseVector &u, const DenseVector &Du) const 
    {
        init(MA) =
            2.*Du(0), 0.,
            2.*u(0), 0.;
        return MA;
    }

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
        init(MB) = 
            2.*u(0), 0.,
            2.*Du(0), -1.;
		return MB;
	}

	const DenseVector &initial() const
    {
        init(init_y) = 0., 0.;
        return init_y;
    }

	const DenseVector &initial_deriv() const
    {
        init(init_yprime) = 1., 2.;
        return init_yprime;
    }
};


/*
Reference:
W. C. Rheinboldt, University of Pittsburgh
sample programs for dae_solve.tgz
http://www.netlib.org/ode/daesolve/

!-------------------------------------------------------------------
!  Driver for applying the DAE-solver with daeq2 to the 
!  Ascher-Petzold problem of index two
!
!  u1' - a*(2 - t) w = [a - 1/(2-t)]*u1 + [(3-t)/(2-t)]*exp(t)
!  u2' -   (a - 1) w = [(a - 1)*u1]/(2-t) - u2 + 2*exp(t)
!  0 = (2 + t)*u1 + (t^2 - 4)*u2 - (t^2 + t - 2)*exp(t)
!
!  with a = 50 and the initial conditions
!
!  u1(0) = 1, u2(0) = 1, w(t0) = -1/2
!
!  Reference: U. Ascher and L. Petzold
!             Lawrence Livermore Nat'l Lab.
!             Num. Math. Group, Tech. Rep. UCRL-JC-107441, 1991
!
!  The exact solution is
!
!  u1(t) = exp[t], u2(t) = exp[t], w = -exp[t]/(2-t)  
!------------------------------------------------------------------
*/
struct ProblemAscherPetzold : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector res;
    mutable DenseVector init_y;

	ProblemAscherPetzold() : ProblemBase(3,3, 1.)
	{
		assert(n == m);
		res.resize(n);
        init_y.resize(n);

        init(MA) =
            1.,0.,0.,
            0.,1.,0.,
            0.,0.,0.;
	}

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
        using utility::sqr;
        const Real a = 50.;
		init(res) =
            Du(0) - a*(2.-t)*u(2) - (a-1./(2.-t))*u(0) - (3.-t)/(2.-t)*exp(t),
            Du(1) - (a-1.)*u(2) - (a-1.)*u(0)/(2.-t) + u(1) - 2.*exp(t),
            (2.+t)*u(0) + (sqr(t) - 4.)*u(1) - (sqr(t) + t - 2.)*exp(t);
		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
        using utility::sqr;
        const Real a = 50.;

        init(MB) = 
            -(a-1./(2.-t)), 0., -a*(2.-t),
            -(a-1.)/(2.-t), 1., -(a-1.),
            (2.+t), (sqr(t)-4.), 0.;
		return MB;
	}

	const DenseVector &initial() const
    {
        init(init_y) = 1., 1., -0.5;
        return init_y;
    }
};



struct ProblemLorenz : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	mutable DenseVector res;

	ProblemLorenz(const Real &T = 800.) : ProblemBase(3,3, T)
	{
		res.resize(3);

		init(MA) = 
			-1, 0, 0,
			0, -1, 0,
			0, 0, -1;
	}

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
        using utility::sqr;
		init(res) =
			10.*(u(1) - u(0)) - Du(0),
			28.*u(0) - u(1) - u(0)*u(2) - Du(1),
			u(0)*u(1) - 8./3.*u(2) - Du(2);
		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
        using utility::sqr;
        init(MB) = 
			-10., 10, 0.,
			28.-u(2), -1., -u(0),
			u(1), u(0), -8./3.;
		return MB;
	}

};

struct ProblemLinearDimBig : public ProblemBase
{
	typedef problem_linear problem_type;


	ProblemLinearDimBig() : ProblemBase(13,13, 0.1) 
	{
//		init(MA) = 1.5742525829456896415, -1.3056437021593288831, -0.39027060117116927996, -1.1450426551690277846, -0.34342044223321826790, -1.0458315759085607014, -0.98879017911691642252, -1.5434002327751127441, -0.54921842400797589827, 1.2750065132525382287, -0.65391283199123695996, 0.44235445896464172663, -0.41650929331756040725, -0.83192099372897300813, 0.72689244975375331358, 0.64408595166906981937, 0.75658270685934991617, -0.057583932091709843052, 0.89483170067599879471, 0.21971041096397701355, 1.2176253672594317522, 0.68532227251471890283, -0.86786097001570731287, 0.39237806610904782412, -0.21967841849908455116, 0.14578073578395513121, 0.32625388491695883121, -0.94696211133104227240, -1.3944587815816239512, -1.8341019518949502728, 1.6097265679766053325, -0.26015212298102844193, 1.9702549612544024040, -2.8455902680543174996, -1.4561869945722282317, -0.18641590481679498483, -0.42620283941738952823, -2.1518348306605983188, 0.35132399680058127625, -1.0023448628596369088, 0.82742836363054705577, -0.31765615147642760678, 0.23321623765523082401, -0.0044864377496023742257, -0.068067408806910744281, 0.33763882679284097226, 0.063106118122364035678, 0.23169141876841351297, -0.41315389163343226217, 0.73321530949813786896, -0.32258704309625712625, -0.15115807212921106250, -2.2140855995637736989, 2.0486619690264454641, 0.70777915971666528590, 1.7130440450726181395, 0.0095446317951120057550, 1.4606576253934419629, 0.13703998642406521790, 2.9696109906569934903, 1.0911045737138595684, -2.2399801979289288928, 0.65644460810810807654, -0.54887626481247982536, 0.56644850327673970393, 0.30168546956383147144, 0.38530904694291283337, 0.45559976703570230037, 0.74518200412343677682, -1.4086322249480976059, -0.33324695978783776972, -1.9274878233616114650, 1.6943199310445463207, 1.0838343460622704768, 0.96035288213380351369, -0.16849360311439967360, 2.0851800140992936379, -0.13842517786544133912, -0.90878660908049463974, 1.1583557235700117063, 0.72110080583171023715, 0.36484668172870155827, -0.36997922662944096095, 0.74101828102441598575, -0.29348820203666248430, 1.6981258469272364286, 1.2648336460628715410, -0.30722817212014770454, 0.10465792018345915364, 0.92269620257598381627, -0.25279750105637830435, -0.86415729318641967903, 0.91613491755200792035, 0.31353384819633222892, 1.4293727829690015991, -0.79891901380658422064, -0.011885299555621512068, -0.44889671602177329905, 1.4960102478239004281, 0.65482373340990650372, -0.17485029590250261250, 0.74749438761726199551, 0.71211192849759045596, 0.092113762811262620348, 0.41831547007975864359, -0.35269364611784683687, -0.26590104688121971648, -0.35662947431494104852, 0.39727685067252936767, -0.10875587045036416037, -0.050730196371360677137, -0.63163360106647654009, -0.41089047033136277563, 0.072496160123933261672, -0.095029350863937498180, -0.23094000639740972550, -0.076221965220509956555, 1.4195722271732858512, -1.3089084674834726076, 0.50367229441706644908, -0.68651404634284780353, 0.0025088486580497593991, -0.22088385703267373595, 0.80773215290104468186, -1.9094585122880085797, -1.1569677929777219450, 1.2257296034674362100, -0.46480133453374294555, -0.43026402900253791152, -0.083703899832320554688, 1.8016903033183840581, -2.5950742302105731452, -1.4027899169894658785, -0.99697039382721218270, 1.3353540553922189254, -0.83873034625325307240, 1.0313561396926817524, -2.9889110404977920593, -1.4800536707409516551, 0.53536482017755258253, -0.16148664032638023489, -1.1125514445244221035, 0.31610786642291605540, 1.3721530919366356663, -0.96686345875484428335, 0.38560620860217277840, -0.43797331214415395073, -1.0221572975454312330, -0.83686803914020314102, -0.49595225684914485275, -1.0764455224370355646, -0.59637798748676814953, 1.6956954877099161356, -0.59367393672683468738, 0.58466830304528105782, -0.14130612102220741847, 1.4342391584756269508, -1.7110340799992620367, -1.0880407921236361476, -2.1887583378607660631, 0.45111871945386025718, -0.90758183525247191725, 0.82242650035569754522, -3.0606878540613270023, -1.0732078542970122322, 1.2137052325538211305, -0.91011314561859390017, -0.97844265644968050925, 0.085416524016352756048;
//		init(MB) = -2.6527224949482949610, 1.8788864799403181340, -2.9771863852689659293, -0.13672920720469107498, -3.6929815431808904698, -0.83918656320018040074, -8.1874001969225362693, 7.5767827483668140928, 6.8308261060698869990, -3.9675320889003410335, -2.4537992656623420337, 0.27409479115748992278, 2.6850456640569543958, 1.3597585215898956773, -1.4202192616592432584, 2.5714907709495937271, 0.42316187240614826408, 2.1302777694681007246, 0.58111773072100021605, 4.9865156073940543709, -4.4724970501486304905, -3.4082450649261331003, 2.5735188955588213093, 1.7389373712828538258, -0.26273939216666717034, -1.7981432852334920336, 1.7493527311380259021, -3.5944331627284503983, -3.3098435691020185821, 2.9178779555157361256, -4.8177634343121599123, -3.4622823703878008337, -9.7981473185001835942, 6.3614708148058164396, 5.1792833843850731316, -0.69637868884162000073, -1.3168911751590631129, 4.5124120803128436153, 2.2855538257132134220, 1.7522107357165705770, -1.8903791736943730479, 1.0748894038737046154, 1.1195730176929011034, 1.5813865762252412825, 0.59982973487073187415, 1.9970708128502896562, -1.4766878933955999705, -1.3605820875098735268, 0.89573564726685197367, 0.77861950578753066640, 0.18960534764898521544, -0.21293402684223877861, 2.7308189426599779540, -1.9816088949672639670, 3.9268556884130617800, 0.34967350832635799747, 4.8059601809115512889, 0.81558726753265796908, 10.389717405055332790, -10.124632632942104128, -9.3252981835742500428, 4.5305131398607496893, 2.9922361994349824458, -2.5139609579338021907, -2.8776431313397892788, -1.8638951232134105492, 3.1477649318572685346, 3.0735679980094311356, -2.8114061620798580132, 3.3332719105060456333, 4.0900620446191303647, 7.6194120205198141262, -3.7496761946942232094, -2.3929528380881263321, -0.70419001615168777971, 0.28519304981772655098, -4.4314075881816736920, -1.3398797543126386642, 1.4772079997025529117, -0.33891643352558873196, 3.0378169182585106460, -1.2605003777672978149, 3.0859234039247823514, 0.85318924653473816185, 7.1576659466966324006, -6.1419610889454322435, -6.3426946383840416604, 2.9466714912156250931, 0.31159385319373636717, -1.1127957190962624131, -2.2511067732132712564, -0.27827721871096659765, 0.99456294973497669573, 2.4894533437676777218, -1.9543905315061572956, 3.0306898998607926370, 2.2128962613258742511, 6.9561437750501691770, -5.4284965980539461587, -3.3544534458166087849, 1.4240986685191059370, 1.0303220932957019473, -3.4049670415227719729, -1.3829129612715122509, 0.045887490518715867824, -0.83775364085675894829, -0.36772072769996984003, 1.6876227823895143667, -0.99844390969248540243, -0.64868226557581861803, -1.1758672946920109995, 1.3207142817439719986, 1.7200426595837859257, -0.27507613343832621176, 0.73191195671250307223, 0.66400543309089296315, 0.39018622385089077251, -0.83580915111525127405, 0.00084121878491870077821, -0.48374391747147706160, 1.0055801695052375100, -2.4882763565754441123, -0.81873235922581112977, -4.4459277418081032169, 4.8770458886561035497, 4.0009930224758938615, -1.2724529769900391274, -1.1217687130133691270, 2.1921218461581797679, 1.0752516553497205675, -2.0599215835161122642, -0.27674638664825555659, -5.7155567078753245582, 1.7062104266735683859, -6.1184752530638358635, -1.9792089733782156123, -13.385799934614878801, 10.765428104220782426, 10.188452314166090011, -4.9399087768931309819, -2.2218713907159239270, 1.5514245476959009573, 4.6877576384454624670, -3.4592718745902541630, 3.3528305273385067452, -1.0828795894999753618, -1.3377003729066475220, -1.3728726986984788724, 0.69796025754845834648, -3.1633101948529045285, 4.7976221190041018406, 4.3393617231085642596, -3.2206819433645572427, -1.4248035993802295669, -0.29294210472809155870, 1.0249174531142652926, -1.6046661223433948341, 0.027053114667658924246, -5.0592408251729822619, 0.66212520523374206928, -4.9304559400902143134, -1.8936590617534891405, -11.109236960698746069, 9.6894472575303320112, 8.3395469452211682275, -3.8133388762711167890, -2.5833879920189675678, 3.4333575331475613659, 3.5176959741622521298;

        init(MA) = 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.;
        init(MB) = 1., 3., 0., -1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 2., 1., 0., 3., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., -2., -5., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 0., 1.;
    }

};


struct ProblemLinearDim : public ProblemBase
{
	typedef problem_linear problem_type;

    enum Type
    {
        INDEX3,
        INDEX2,
        INDEX1,
        INDEX0,
    };


	ProblemLinearDim(Type type) : ProblemBase(3, 3, 1.) 
	{

        switch (type)
        {
        case INDEX3:
//          A := Matrix(3,3, [[0,1,0],[0,0,1],[0,0,0]]):
//          B := Matrix(3,3, [[1,0,0],[0,1,0],[0,0,1]]):

            init(MA) = -1.6326019600313520850, -0.86805646882238552894, 0.15160170928481124747, -1.9714043735106444537, -1.0763163755025890019, 0.56282428278598695066, -0.51001846406346166997, -0.30294741161184435478, 0.47644122273131368392;
            init(MB) = -0.51630290883522094628, -0.036132050492988740954, 0.65579338485034990540, -1.2055490384834568655, -0.33111519085669396823, 0.33000524288221751123, -0.97483416535594611229, -0.46597825343636736648, 0.33235314231005949532;
            break;
        case INDEX2:
//          A := Matrix(3,3, [[1,0,0],[0,0,1],[0,0,0]]):
//          B := Matrix(3,3, [[-1,0,0],[0,1,0],[0,0,1]]):

//          init(MA) = 1,0,0, 0,0,1, 0,0,0;
//          init(MB) = -1,0,0, 0,1,0, 0,0,1;           

            init(MA) = -1.3106550463874051167, 0.75222769554216425664, 0.29178227866668296439, -0.82400899634523474838, -0.64898974465536419308, -0.31706005304910097664, 1.8909193140286758504, -0.34848003324736282407, -0.092273655131950471208;
            init(MB) = -0.84110633044456124482, -1.3718387502903032600, -0.97805307354756695310, 0.35559704685189033260, -0.36780506288885357715, -0.74560266956766205034, -0.75537213822102162301, 1.6244820378677683930, 1.4371967094889440295;
            break;
        case INDEX1:
//          A := Matrix(3,3, [[1,0,0],[0,1,0],[0,0,0]]):
//          B := Matrix(3,3, [[0,1,0],[1,0,0],[0,0,1]]):

            init(MA) = 0.23379773265636110822, -0.050421899840117398216, 0.54579753336926540059, 0.54902689017016400422, -0.088184539850625263263, 1.3116261374084524904, -0.44406147101052145280, 0.97218133314008524053, -0.16866935100345581941; 
            init(MB) = 0.38513363558542848990, -0.15027379329398922944, 0.24320272479810858429, 0.44339114168859579595, 0.34373853533043159244, 1.2459157468510374252, -0.10101138201071621065, -0.51241694483823775856, 0.10736592281248443070;
            break;

        case INDEX0:
//          A := Matrix(3,3, [[1,0,0],[0,1,0],[0,0,1]]):
//          B := Matrix(3,3, [[1,2,0],[-2,1,3],[-1,0,1]]):

            init(MA) = 0.47735373912413321436, -0.41575590954177555744, 0.21525723887728897082, 0.64173162369161227790, -0.22724164013279767378, 1.1858125704864869761, -0.80454465163256894654, 1.5129061040731564811, 0.32055782269789435066;
            init(MB) = 0.54716763215225534112, 0.22803878851562818463, 1.5073781517933573988, 0.44180137417510994760, -0.79318075792505899351, 1.2233238370012727199, 0.51050560042031460186, 0.035042824079316264092, 0.95992064750189096278;
            break;
        default:
            throw "ProblemLinearDim: Unknown problem type.";
        }



/*
        assert(n == m);
        for (int i=0; i<n-1; i++)
            MA(i,i+1) = 1.;
        for (int i=0; i<n; i++)
            MB(i,i) = 1.;
*/            
            
/*
        // 1-dim solution space
        init(MA) = -7668,   -2250,   -1260,
                   -2241,    -666,    -372,
                    -159,   -5332,   -2384;
        MA /= 11178.;

        init(MB) = -8961,   -3208,   -5966,
                    2754,    -810,   -1510,
                  -12909,    7186,     892;
        MB /= 20987.;
*/
/*
        init(MA) = -8478,   -2142,   -2394,
                   -2481,    -634,    -708,
                   -2229,   -5056,   -5282;
        MA /= 13014;

        init(MB) = -4344,   -9364,   -8558,
                    4122,   -2634,   -2278,
                   -1110,   -8546,   -5732;
        MB /= 22266;
*/
/*
        init(MA) = 0, 1, 0,
                   0, 0, 1,
                   0, 0, 0;
        init(MB) = 1, 0, 0,
                   0, 1, 0,
                   0, 0, 1;

*/                   


//        init(MA) = 3., 0., -2., -2.600000000, -0.4000000000, 1.800000000, 1.200000000, -1.200000000, -0.6000000000;
//        init(MB) = 4., 2., -3., -2.200000000, -1.600000000, 2.200000000, 2.400000000, 1.200000000, -0.4000000000;

/*
        init(MA) = 2.022386987, 2.293057230, 1.322573800, 4.667082872,
         -1.541504487, 2.589548050, -0.3518405594, -0.7189880042,
         -4.718301989, -0.5599828688, -1.623106135, -1.713549366,
         -0.5723420475, -2.275500071, 1.115219233, -1.796328210,
         -0.2382986102, 0.3366779572, 2.486943527, 0.1630489176,
         -1.771076640, -0.5188879490, -0.1674607727, -3.065887675,
         0.5076410815, -1.173145295, 0.01617154107, 0.3346473082,
         3.329300018, -0.2421673340, -2.759783185, -2.646165272,
         -0.4435401327, -4.385464170, 2.513817649, -2.555676677,
         1.235047885, 1.365529037, 4.940032577, 0.2642175884,
         -0.3631338564, -0.08516627581, 0.3325878391, 0.01790985167,
         0.4135968379, -0.3713133065, 0.4913559551, 0.4865196196,
         -1.016313246, 0.8648741982, 1.395758890, 0.8059609017,
         0.1527642166, 2.643250727, -1.817101260, 1.217734422,
         -1.252688607, -1.456974817, -3.058113064, -0.4620157885,
         0.8059614495, 0.02870831777, 0.1976554148, 1.270225680,
         0.8141147118, 0.2007504645, 0.5070804901, 1.177846943,
         -1.245876708, 0.1857176716, -2.840000037, -3.415769226,
         -1.265312375, -5.361052810, 2.962377306, -2.881375137,
         1.430242204, 1.833511327, 6.030165964, 0.5325056407,
         0.6950392907, -0.8177860401, -0.05966622744, 0.06192208972,
         1.076918022, -0.03474311790, 0.3434557890, 1.173242116,
         -0.6613886720, 0.3779414954, 0.4994424985, -0.5177099269,
         0.09555064619, -1.373468575, 1.191816877, 0.1956694348,
         1.060166908, 1.266668394, 1.538659969, -0.2818942205;
 
        init(MB) = -5.497549974, -4.427317854, -0.02440819895, -2.440903129,
         -0.6014462964, -5.315254895, -1.686111482, -1.642053001,
         -3.594053626, 3.245262670, 1.908162764, 1.652811235,
         -0.8496182351, 0.5438193561, -0.3055405901, 2.096383787,
         0.2449892808, -0.03532631125, 2.637798655, -1.465309424,
         3.726165977, 1.691078319, 0.6482818827, 1.859136813,
         1.182342789, 3.260121241, 1.484631962, 0.2780113606,
         2.015557656, -2.627688667, 7.192663986, 4.116708088,
         0.2831488188, 2.655154487, 1.404362416, 6.361580713,
         1.515595849, 1.533494824, 3.238110351, -3.915900638,
         -1.020066274, -1.310697973, -1.125206011, 0.06431090447,
         -0.8416041323, -0.8466503672, -0.5742433577, -0.9994401510,
         -0.2834135210, 0.5114533069, -3.657740619, -1.200646420,
         0.7219710121, -1.918097107, -0.2855822993, -2.470321222,
         -0.3541260631, 0.3660290304, -1.615256902, 2.044864234,
         -0.1952887881, -0.2780684734, -0.7452864385, -0.9492814867,
         -1.074824381, -0.7526017709, -0.8930097075, -0.1779205900,
         -1.106631518, 1.091924230, 8.037616572, 4.078056784,
         -0.3588023350, 3.075869177, 1.767543348, 6.710742134,
         1.419794010, 1.802833837, 4.775326603, -4.807726429,
         -0.9048370111, 0.1375521436, -0.8822682095, -0.3826252619,
         -1.167079098, -0.5509381385, -0.5916995117, -0.6070917220,
         0.4554663049, 0.5982308590, 2.911538338, 2.384957433,
         0.5014221164, 1.262175749, -0.01299044657, 2.593663927,
         -0.1023102985, 0.4720935910, 1.033955026, -1.932192957;
*/         

/*
        init(MA) = -7,   -3,    5,
            -2,   -6,    2,
            12,    9,   -9;


        init(MB) = 0,   -6,    3,
             7,    0,    0,
            -6,    9,   -4;
*/
	}

	const DenseVector &g(const Real &t) const 
    { 
        init(vg) = sin(t), cos(t), exp(t);
        return vg; 
    }
};

// Exercise 16 from Chapter 10, Ascher, Petzold - Computer Methods for ODE abd DAE.
// TODO: analyze problems
struct ProblemSonic : public ProblemBase
{
	typedef problem_nonlinear problem_type;

	enum IndexType
	{
		PHI = 0,
		RHO,
		E
	};

	mutable DenseVector res;

	const Real J;
	const Real alpha;
	const Real rho_bar;
	const Real b;
	ProblemSonic(const Real &J=0.5, const Real &alpha=0.,
			const Real &rho_bar=3., const Real &b=10.3)
			: ProblemBase(3,3, 20.), J(J), alpha(alpha), rho_bar(rho_bar), b(b)
	{
		res.resize(3);

		MA(0, PHI) = 1.;
		MA(1, E) = 1.;
	}

	const DenseVector &F(const Real &t, const DenseVector &u, const DenseVector &Du) const
	{
		using utility::sqr;

		init(res) = 
			Du[PHI] - (u[RHO]*u[E] - alpha*J),
			Du[E] - (u[RHO] - 1.),
			u[PHI] - (J*J/u[RHO] + u[RHO]);

		return res;
	}

	const DenseMatrix &B(const Real &t, const DenseVector &u, const DenseVector &Du) const 
	{
		MB(0, RHO) = -u[E];
		MB(0, E) = -u[RHO];

		MB(1, RHO) = -1.;

		MB(2, PHI) = 1.;
		MB(2, RHO) = J*J/(u[RHO]*u[RHO]) - 1.;

		return MB;
	}
};


} // end namespace problems

#endif // __PROBLEMS_HXX__
