/*
    This file is part of SGDAE, a software to numerically solve differential
    algebraic equations using a steepest descent method based on Sobolev 
    gradients.
    You are welcome to contact the authors via e-mail:
        <manfred-sauter [at] gmx [dot] de>
        <robin.nittka [at] gmx [dot] de>

    Copyright 2005-2008 Manfred Sauter, Robin Nittka.

    SGDAE is free software distributed under the terms of the revised BSD 
    license as illustrated on <http://creativecommons.org/licenses/BSD/>.
    For details consult the accompanying LICENSE.txt file.

    $Id: $
*/


#ifndef __UTILITY_HEADER__
#define __UTILITY_HEADER__

#include <limits>


namespace utility {
// #TODO: Ensure support for mpz_float of the gmp library.

template<typename T> struct PrecInfo;

// Implementation of precision info type for standard double numbers.
template<> struct PrecInfo<double>
{
	static const double EPS;
	static const double MACHINE_EPS;
};
const double PrecInfo<double>::MACHINE_EPS = std::numeric_limits<double>::epsilon();
const double PrecInfo<double>::EPS = std::numeric_limits<double>::epsilon();

#ifdef GMP

// GNU multiprecision library.
#include <gmp.h>
#include <gmpxx.h>

// Specialization of precision info for the gmpxx multi precision floats.
template<> struct PrecInfo<mpf_class>
{
	static const int BITS;
	static const mpf_class EPS;
	static const mpf_class MACHINE_EPS;

	static int init(int bits)
	{
		mpf_set_default_prec(bits);
		return bits;
	}
};
const int PrecInfo<mpf_class>::BITS = PrecInfo<mpf_class>::init(512);
const mpf_class PrecInfo<mpf_class>::MACHINE_EPS = mpf_class(PrecInfo<double>::MACHINE_EPS);
const mpf_class PrecInfo<mpf_class>::EPS = mpf_class("1e-13");

#endif // GMP

inline double sqr(const double &a)
{
	return a*a;
}


// BUG: Somewhere inside of the Pardiso solver srand is called with a constant. Therefore the random numbers repeat after each call of the solver.
// Workaround: We use the following example implementation of rand/srand.

/*
    POSIX.1-2001 gives the following example of an implementation of rand() and srand(), possibly useful when
    one needs the same sequence on two different machines.
*/

static unsigned long next = 1;
const unsigned int rand_max = 32767;

/* RAND_MAX assumed to be 32767 */
int rand(void)
{
    next = next * 1103515245 + 12345;
    return((unsigned)(next/65536) % 32768);
}

void srand(unsigned seed)
{
    next = seed;
}

double rand_uniform(const double &a=0., const double &b=1.)
{
    return a + double(rand())/rand_max * (b-a);
}

inline double sign(const double &a)
{
	if (a < 0.) return -1.;
	if (a > 0.) return 1.;

	return 0.;	
}


inline double flabs(const double &a)
{
	return fabs(a);
}

inline bool is_zero(const double &a)
{
	return flabs(a) < PrecInfo<double>::MACHINE_EPS;
}



} // end namespace utility


// addons
#include <boost/assign/list_inserter.hpp>
#include <boost/timer.hpp>

namespace boost { namespace assign {

template<typename T>	
class DataInitializer
{
public:
	typedef typename T::value_type value_type;
	DataInitializer(T &A) { ptr = &A.data()[0]; }

	void push_back(value_type v) { *ptr++ = v; }

	template<typename V>
	inline list_inserter<assign_detail::call_push_back<DataInitializer>, value_type> 
	operator =(V v) { return assign::push_back(*this)(v); }   
private:	
	value_type *ptr;
};

template<typename T> DataInitializer<T> init(T &obj)
{
	DataInitializer<T> myinit(obj);
	return myinit;
}

}}

struct InvalidInput : public std::exception
{
	const char *msg;

	InvalidInput(const char *message="Invalid input.") : msg(message) {}
	virtual const char* what() const throw() { return msg; }
};

struct MinimalStepSize : public std::exception
{
	const char *msg;

	MinimalStepSize(const char *message="Minimal step size reached.") : msg(message) {}
	virtual const char* what() const throw() { return msg; }
};

struct MaximumPrecisionReached : public std::exception
{
	const char *msg;

	MaximumPrecisionReached(const char *message="Maximum precision was reached.") : msg(message) {}
	virtual const char* what() const throw() { return msg; }
};

#endif // __UTILITY_HEADER__
